package io.magus.methodmap.domain;

import io.magus.methodmap.AbstractMethodMap;
import io.magus.methodmap.ArgsFactory;
import io.magus.methodmap.AutoInstantiatingMethodMap;
import io.magus.methodmap.KeyArgsPair;
import io.magus.methodmap.domain.error.UnmappedDomainException;
import io.magus.methodmap.error.ArgsProductionException;
import io.magus.methodmap.error.InvocationException;
import io.magus.methodmap.error.UnmappedKeyException;

import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Objects;

/**
 * Instances of this class can contain subclasses of
 * {@link io.magus.methodmap.AbstractMethodMap}.
 * The purpose of this class is to abstract away the process of invoking underlying instances of
 * {@link io.magus.methodmap.AutoInstantiatingMethodMap} and other
 * {@link io.magus.methodmap.AbstractMethodMap} implementations via a common interface.
 *
 * @author Enseart A. Simpson
 *
 * @param <K>	Type of the keys contained by the MethodMapDomain
 * @param <V>	Type of the keys contained by all contained methodmaps
 */
public class MethodMapDomain<K, V> extends HashMap<K, AbstractMethodMap<V>> {

	private static final long serialVersionUID = -7173271692302582974L;

	public MethodMapDomain() {
        super();
    }

	/**
	 * Retrieves the methodmap mapped to the provided domain.
	 *
	 * @param domain					the domain to which the desired methodmap is mapped
	 * @return							the methodmap mapped to the provided domain
	 * @throws UnmappedDomainException	If there is no methodmap mapped to the provided domain.
	 */
	public AbstractMethodMap<V> getMethodMap(K domain) throws UnmappedDomainException {
		AbstractMethodMap<V> methodMap;

		methodMap = get(domain);

		if(methodMap == null)
			throw new UnmappedDomainException(Objects.toString(domain));

		return methodMap;
	}

	/**
	 * Retrieves the methodmap mapped to the provided domain and calls {@code invoke} with
	 * the given arguments. <p>If the methodmap is an instance of
	 * {@link io.magus.methodmap.AutoInstantiatingMethodMap} then
	 * {@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
	 * is invoked with {@code key}, {@code constructorParameterTypes}, {@code constructorArgs},
	 * and {@code args}. <p>Otherwise
	 * {@link io.magus.methodmap.AbstractMethodMap#invoke(Object, Object...)}
	 * is invoked with {@code key} and {@code args}.
	 *
	 * @param domain						the domain to which the desired methodmap is mapped
	 * @param key							the key to which the desired methodhandleinfo is mapped
	 * @param constructorParameterTypes		the parameter types of the desired constructor to use if the
	 * 										methodmap is an instance of
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap}
	 * @param constructorArgs				the arguments to be passed to the desired constructor if
	 * 										the methodmap is an instance of
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap}
	 * @param args							the arguments to be passed to the methodhandle
	 * @return								the result of invoking the methodmap with the
	 * 										provided arguments
	 * @throws UnmappedDomainException		If there is no methodmap mapped to the provided domain.
	 * @throws NoSuchMethodException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
	 * @throws SecurityException			See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
	 * @throws InstantiationException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
	 * @throws IllegalAccessException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
	 * @throws IllegalArgumentException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
	 * @throws InvocationTargetException	See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
	 * @throws UnmappedKeyException			If there is no methodhandleinfo associated with the provided key.
	 * @throws InvocationException			If methodhandle invocation fails.
	 */
    public Object invoke(K domain, V key, Class<?>[] constructorParameterTypes,
			Object[] constructorArgs, Object... args)
    			throws UnmappedDomainException, NoSuchMethodException, SecurityException,
    			InstantiationException, IllegalAccessException, IllegalArgumentException,
    			InvocationTargetException, UnmappedKeyException, InvocationException {
		AbstractMethodMap<V> methodMap;

		methodMap = getMethodMap(domain);

		return (methodMap instanceof AutoInstantiatingMethodMap) ?
				((AutoInstantiatingMethodMap<V>)methodMap)
					.invoke(key, constructorParameterTypes, constructorArgs, args) :
				methodMap.invoke(key, args);
    }

    /**
     * Retrieves the methodmap mapped to the provided domain and calls {@code invoke} with
	 * the given argsfactory. <p>If the methodmap is an instance of
	 * {@link io.magus.methodmap.AutoInstantiatingMethodMap} then
	 * {@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], ArgsFactory)}
	 * is invoked with {@code key}, {@code constructorParameterTypes}, {@code constructorArgs},
	 * and {@code argsFactory}. <p>Otherwise
	 * {@link io.magus.methodmap.AbstractMethodMap#invoke(Object, ArgsFactory)}
	 * is invoked with {@code key} and {@code argsFactory}.
     *
     * @param domain						the domain to which the desired methodmap is mapped
	 * @param key							the key to which the desired methodhandleinfo is mapped
	 * @param constructorParameterTypes		the parameter types of the desired constructor to use if
	 * 										the methodmap is an instance of
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap}
	 * @param constructorArgs				the arguments to be passed to the desired constructor if
	 * 										the methodmap is an instance of
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap}
     * @param argsFactory					Produces the arguments that are passed to the
     * 										methodhandle.
     * @return								the result of invoking the methodhandle with the
     * 										arguments produced by the provided argsfactory
     * @throws UnmappedDomainException		If there is no methodmap mapped to the provided domain.
     * @throws NoSuchMethodException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], ArgsFactory)}
	 * @throws SecurityException			See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], ArgsFactory)}
	 * @throws InstantiationException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], ArgsFactory)}
	 * @throws IllegalAccessException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], ArgsFactory)}
	 * @throws IllegalArgumentException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], ArgsFactory)}
	 * @throws InvocationTargetException	See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], ArgsFactory)}
     * @throws ArgsProductionException		If the argsfactory fails to produce arguments.
     * @throws UnmappedKeyException			If there is no methodhandleinfo associated with the
     * 										provided key.
	 * @throws InvocationException			If methodhandle invocation fails.
     */
    public Object invoke(K domain, V key, Class<?>[] constructorParameterTypes,
            Object[] constructorArgs, ArgsFactory argsFactory)
                    throws UnmappedDomainException, NoSuchMethodException, SecurityException,
                    InstantiationException, IllegalAccessException, IllegalArgumentException,
                    InvocationTargetException, ArgsProductionException, UnmappedKeyException,
                    InvocationException {
        AbstractMethodMap<V> methodMap;

        methodMap = getMethodMap(domain);

        return (methodMap instanceof AutoInstantiatingMethodMap) ?
                ((AutoInstantiatingMethodMap<V>)methodMap)
                    .invoke(key, constructorParameterTypes, constructorArgs, argsFactory) :
                methodMap.invoke(key, argsFactory);
    }

    /**
     * Convenience method for calling
     * {@link #invoke(Object, Object, Class[], Object[], Object...) invoke(Object, Object, null, null, Object...)}
     *
     * @param domain						the domain to which the desired methodmap is mapped
	 * @param key							the key to which the desired methodhandleinfo is mapped
     * @param args							the arguments to be passed to the methodhandle
     * @return								the result of invoking the methodhandle with the
     * 										provided arguments
     * @throws NoSuchMethodException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
	 * @throws SecurityException			See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
	 * @throws InstantiationException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
	 * @throws IllegalAccessException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
	 * @throws IllegalArgumentException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
	 * @throws InvocationTargetException	See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
     * @throws UnmappedDomainException		If there is no methodmap mapped to the provided domain.
     * @throws UnmappedKeyException			If there is no methodhandleinfo associated with the
     * 										provided key.
	 * @throws InvocationException			If methodhandle invocation fails.
     */
    public Object invoke(K domain, V key, Object... args)
            throws NoSuchMethodException, SecurityException, InstantiationException,
            IllegalAccessException, IllegalArgumentException, InvocationTargetException,
            UnmappedDomainException, UnmappedKeyException, InvocationException {
        return invoke(domain, key, null, null, args);
    }

    /**
     * Convenience method for calling {@link #invoke(Object, Object, Class[], Object[], ArgsFactory) invoke(Object, Object, null, null, ArgsFactory)}
     *
     * @param domain						the domain to which the desired methodmap is mapped
	 * @param key							the key to which the desired methodhandleinfo is mapped
     * @param argsFactory					Produces the arguments that are passed to the methodhandle.
     * @return								the result of invoking the methodhandle with the provided argsfactory
     * @throws NoSuchMethodException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], ArgsFactory)}
	 * @throws SecurityException			See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], ArgsFactory)}
	 * @throws InstantiationException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], ArgsFactory)}
	 * @throws IllegalAccessException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], ArgsFactory)}
	 * @throws IllegalArgumentException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], ArgsFactory)}
	 * @throws InvocationTargetException	See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], ArgsFactory)}
     * @throws UnmappedDomainException		If there is no methodmap mapped to the provided domain.
     * @throws ArgsProductionException		If the argsfactory fails to produce arguments.
     * @throws UnmappedKeyException			If there is no methodhandleinfo associated with the provided key.
	 * @throws InvocationException			If methodhandle invocation fails.
     */
    public Object invoke(K domain, V key, ArgsFactory argsFactory)
            throws NoSuchMethodException, SecurityException, InstantiationException,
            IllegalAccessException, IllegalArgumentException, InvocationTargetException,
            UnmappedDomainException, ArgsProductionException, UnmappedKeyException,
            InvocationException {
        return invoke(domain, key, null, null, argsFactory);
    }

    /**
     * Treats the first argument as the domain, the second argument as the key, and the remaining
     * arguments as the methodhandle arguments and calls
     * {@link #invoke(Object, Object, Object...) invoke(Object, Object, Object...)}.
     *
     * @param args							the arguments to be passed to the methodhandle
     * @return								the result of invoking the methodhandle with the provided arguments
     * @throws NoSuchMethodException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
	 * @throws SecurityException			See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
	 * @throws InstantiationException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
	 * @throws IllegalAccessException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
	 * @throws IllegalArgumentException		See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
	 * @throws InvocationTargetException	See the documentation for
	 * 										{@link io.magus.methodmap.AutoInstantiatingMethodMap#invoke(Object, Class[], Object[], Object...)}
     * @throws UnmappedDomainException		If there is no methodmap mapped to the provided domain.
     * @throws UnmappedKeyException			If there is no methodhandleinfo associated with the provided key.
	 * @throws InvocationException			If methodhandle invocation fails.
     */
    public Object invoke(Object... args)
            throws NoSuchMethodException, SecurityException, InstantiationException,
            IllegalAccessException, IllegalArgumentException, InvocationTargetException,
            UnmappedDomainException, UnmappedKeyException, InvocationException {
        KeyArgsPair<K> domainMapArgs;
        KeyArgsPair<V> methodMapArgs;

        domainMapArgs = new KeyArgsPair<K>(args);
        methodMapArgs = new KeyArgsPair<V>(domainMapArgs.getArgs());

        return invoke(domainMapArgs.getKey(), methodMapArgs.getKey(), methodMapArgs.getArgs());
    }
}