package io.magus.methodmap;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Arrays;

/**
 * The entry value for implementations of {@link io.magus.methodmap.AbstractMethodMap}.
 * Contains all the relevant information about the underlying method that is invokable via the
 * contained method or methodhandle.
 *
 * @author Enseart A. Simpson
 *
 */
public class MethodHandleInfo {

	private final Object target;
	private final Method method;
	private final MethodHandles.Lookup lookup;
	private final MethodHandle methodHandle;
	private final int modifiers;

	/**
	 * @param target					the target object to which the methodhandle will be bound if
	 * 									it is non-static
	 * @param method					the java method represented by this methodhandleinfo
	 * @param lookup					the lookup used to produce the methodhandle for this
	 * 									methodhandleinfo
	 * @throws IllegalAccessException	See the documentation for
	 * 									{@link java.lang.invoke.MethodHandles.Lookup#unreflect(Method)}
	 */
	public MethodHandleInfo(Object target, Method method, MethodHandles.Lookup lookup)
			throws IllegalAccessException {
		MethodHandle tempHandle;

		this.target = target;
		this.method = method;
		this.lookup = lookup;

		tempHandle = lookup.unreflect(method);

		modifiers = method.getModifiers();

		if(target != null && !Modifier.isStatic(modifiers))
			tempHandle = tempHandle.bindTo(target);

		if(method.isVarArgs()) {
			Class<?> paramTypes[] = method.getParameterTypes();
			tempHandle = tempHandle.asVarargsCollector(paramTypes[paramTypes.length - 1]);
		}

		methodHandle = tempHandle;
	}

	/**
	 * Retrieves the target object to which the methodhandle is bound if it is non-static.
	 *
	 * @return	the target object
	 */
	public Object getTarget() {
		return target;
	}

	/**
	 * Retrieves the underlying method.
	 *
	 * @return	the underlying method
	 */
	public Method getMethod() {
		return method;
	}

	/**
	 * Retrieves the underlying lookup.
	 *
	 * @return	the underlying lookup
	 */
	public MethodHandles.Lookup getLookup() {
		return lookup;
	}

	/**
	 * Retrieves the underlying methodhandle.
	 *
	 * @return	the underlying methodhandle
	 */
	public MethodHandle getMethodHandle() {
		return methodHandle;
	}

	/**
	 * Retrieves the access modifier value of the underlying method.
	 *
	 * @return	the access modifier value of the underlying method
	 */
	public int getModifiers() {
		return modifiers;
	}

	public Object invoke(Object... args) throws Throwable {
		Object result, target, methodArgs[];

		if(method.isVarArgs()) {
			if(Modifier.isStatic(modifiers)) {
				result = methodHandle.invokeWithArguments(args);
			} else {
				if(args == null) {
					result = methodHandle.invokeWithArguments(args);
				} else {
					target = null;
					methodArgs = null;

					if(args.length >= 1) {
						target = args[0];
					}

					if(args.length > 1) {
						methodArgs = Arrays.copyOfRange(args, 1, args.length);
					}

					result = method.invoke(target, methodArgs);
				}
			}
		} else {
			result = methodHandle.invokeWithArguments(args);
		}

		return result;
	}
}
