package io.magus.methodmap;

import io.magus.methodmap.error.ArgsProductionException;
import io.magus.methodmap.error.DuplicateKeyException;
import io.magus.methodmap.error.InvocationException;
import io.magus.methodmap.error.MapEntryKeyProductionException;
import io.magus.methodmap.error.MethodCollectionProductionException;
import io.magus.methodmap.error.UnmappedKeyException;

import java.lang.invoke.MethodHandles;

/**
 * MethodMap which is bound to a single object.
 * All non-static methods are invoked with the the same object as the target.
 *
 * @author Enseart A. Simpson
 *
 * @param <K>	The type of keys contained by the MethodMap
 */
public class BoundMethodMap<K> extends AbstractMethodMap<K> {

	private static final long serialVersionUID = 1388937976624549693L;

	/**
	 * @param entryKeyFactory						the factory used to generate keys for each of
	 * 												the methods produced by
	 * 												{@code methodCollectionFactory}
	 * @param methodCollectionFactory				Produces the collection of methods that will be
	 * 												included in this methodmap. The target object is
	 * 												also contained in this factory.
	 * @param lookup								the lookup object used to convert the produced
	 * 												methods into methodhandles
	 * @throws DuplicateKeyException				If more than one of the same key is generated
	 * 												for the entries of the methodmap.
	 * @throws IllegalAccessException				If methodhandle lookup fails for any of the
	 * 												produced methods.
	 * @throws MethodCollectionProductionException	If the methodcollectionfactory fails to produce
	 * 												a method collection.
	 * @throws MapEntryKeyProductionException		If entry keys fail to be generated for any of
	 * 												the produced methods.
	 */
	public BoundMethodMap(MapEntryKeyFactory<K> entryKeyFactory,
			AbstractBoundMethodCollectionFactory methodCollectionFactory,
			MethodHandles.Lookup lookup) throws DuplicateKeyException, IllegalAccessException,
			MethodCollectionProductionException, MapEntryKeyProductionException {

		super(methodCollectionFactory.getTarget(),
				methodCollectionFactory.produceMethodCollection(), entryKeyFactory, lookup);
	}

	@Override
	public Object invoke(K key, Object... args) throws UnmappedKeyException, InvocationException {
		return super.invoke(getMethodHandleInfo(key), args);
	}

	@Override
	public Object invoke(K key, ArgsFactory argsFactory)
			throws UnmappedKeyException, ArgsProductionException, InvocationException {
		MethodHandleInfo methodHandleInfo;

		methodHandleInfo = getMethodHandleInfo(key);
		return super.invoke(methodHandleInfo, argsFactory.produceArgs(methodHandleInfo));
	}

	@Override
	public Object invoke(Object... args) throws UnmappedKeyException, InvocationException {
		KeyArgsPair<K> keyArgsPair;

		keyArgsPair = new KeyArgsPair<K>(args);
		return invoke(keyArgsPair.getKey(), keyArgsPair.getArgs());
	}
}
