package io.magus.methodmap;

import io.magus.methodmap.error.ArgsProductionException;
import io.magus.methodmap.error.DuplicateKeyException;
import io.magus.methodmap.error.InvocationException;
import io.magus.methodmap.error.MapEntryKeyProductionException;
import io.magus.methodmap.error.MethodCollectionProductionException;
import io.magus.methodmap.error.UnmappedKeyException;

import java.lang.invoke.MethodHandles;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Modifier;

/**
 * MethodMap that automatically instantiates an object of the appropriate class when invoking
 * non-static methods. Supports the instantiation of objects with specific constructors and
 * arguments if available for the desired class.
 *
 * @author Enseart A. Simpson
 *
 * @param <K>	Type of the keys contained in the MethodMap
 */
public class AutoInstantiatingMethodMap<K> extends AbstractMethodMap<K> {

	private static final long serialVersionUID = 4325779650129661784L;

	/**
	 * @param entryKeyFactory						factory used to generate keys for each of
	 * 												the methods produced by
	 * 												{@code methodCollectionFactory}
	 * @param methodCollectionFactory				Produces a collection of methods that will be
	 * 												included in this methodmap.
	 * @param lookup								the lookup object used to convert the produced
	 * 												methods into methodhandles
	 * @throws DuplicateKeyException				If more than one of the same key is generated
	 * 												for the entries of the methodmap.
	 * @throws MethodCollectionProductionException	If the methodcollectionfactory fails to produce
	 * 												a method collection.
	 * @throws IllegalAccessException				If MethodHandle lookup fails for any of the
	 * 												produced methods.
	 * @throws MapEntryKeyProductionException		If entry keys fail to be generated for any of
	 * 												the produced methods.
	 */
	public AutoInstantiatingMethodMap(MapEntryKeyFactory<K> entryKeyFactory,
			MethodCollectionFactory methodCollectionFactory, MethodHandles.Lookup lookup)
			throws DuplicateKeyException, MethodCollectionProductionException,
			IllegalAccessException, MapEntryKeyProductionException {

		super(null, methodCollectionFactory.produceMethodCollection(), entryKeyFactory, lookup);
	}

	/**
	 * Invokes the methodhandle contained in the provided methodhandleinfo with the given
	 * arguments.
	 * <p> Attempts to instantiate an object with the provided constructor parameter types
	 * and arguments if the underlying method is non-static.
	 *
	 * @param methodHandleInfo				the methodhandleinfo containing the methodhandle to be
	 * 										invoked
	 * @param constructorParameterTypes		the parameter types of the desired constructor
	 * @param constructorArgs				the arguments to be passed to the desired constructor
	 * @param args							the arguments to be passed to the methodhandle
	 * @return								the result of invoking the methodhandle with the provided
	 * 										arguments
	 * @throws SecurityException			See the documentation for
	 * 										{@link java.lang.Class#getConstructor(Class...)}.
	 * @throws NoSuchMethodException		See the documentation for
	 * 										{@link java.lang.Class#getConstructor(Class...)}.
	 * @throws InvocationTargetException	See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws IllegalArgumentException		See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws IllegalAccessException		See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws InstantiationException		See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws InvocationException          If methodhandle invocation fails.
	 *
	 */
	protected Object invoke(MethodHandleInfo methodHandleInfo, Class<?>[] constructorParameterTypes,
			Object[] constructorArgs, Object... args)
			throws NoSuchMethodException, SecurityException, InstantiationException,
			IllegalAccessException, IllegalArgumentException, InvocationTargetException,
			InvocationException {

		Constructor<?> constructor;
		Object instance, invokeArgs[];
		int argsLength, i;

		if(!(Modifier.isStatic(methodHandleInfo.getModifiers()))) {
			constructor = methodHandleInfo.getMethod()
					.getDeclaringClass().getConstructor(constructorParameterTypes);

			instance = constructor.newInstance(constructorArgs);

			argsLength = (args == null) ? 0 : args.length;
			invokeArgs = new Object[argsLength + 1];

			i = 0;
			invokeArgs[i++] = instance;

			if(args != null)
				for(Object arg : args)
					invokeArgs[i++] = arg;

		} else {
			invokeArgs = args;
		}

		return super.invoke(methodHandleInfo, invokeArgs);
	}

	/**
	 * Invokes the methodhandle contained in the provided methodhandleinfo with the arguments
	 * produced by the given argsfactory.
	 * <p> Attempts to instantiate an object with the provided
	 * constructor parameter types and arguments if the underlying method is non-static.
	 *
	 * @param methodHandleInfo				the methodhandleinfo containing the methodhandle to
	 * 										be invoked
	 * @param constructorParameterTypes		the parameter types of the desired constructor
	 * @param constructorArgs				the arguments to be passed to the desired constructor
	 * @param argsFactory					Produces the arguments that are passed to the methodhandle.
	 * @return								the result of invoking the methodhandle with the produced
	 * 										arguments
	 * @throws InvocationTargetException	See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws IllegalArgumentException		See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws IllegalAccessException		See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws InstantiationException		See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws SecurityException			See the documentation for
	 * 										{@link java.lang.Class#getConstructor(Class...)}.
	 * @throws NoSuchMethodException		See the documentation for
	 * 										{@link java.lang.Class#getConstructor(Class...)}.
	 * @throws ArgsProductionException		If the argsfactory fails to produce arguments.
	 * @throws InvocationException          If methodhandle invocation fails.
	 *
	 */
	protected Object invoke(MethodHandleInfo methodHandleInfo, Class<?>[] constructorParameterTypes,
			Object[] constructorArgs, ArgsFactory argsFactory)
			throws NoSuchMethodException, SecurityException, InstantiationException,
			IllegalAccessException, IllegalArgumentException, InvocationTargetException,
			ArgsProductionException, InvocationException {

		return invoke(methodHandleInfo, constructorParameterTypes,
				constructorArgs, argsFactory.produceArgs(methodHandleInfo));
	}

	/**
	 * {@inheritDoc}<p>
	 * Convenience method for instantiating objects with the default constructor when the underlying
	 * method is non-static.<p>
	 * Equivalent to calling {@link #invoke(MethodHandleInfo, Class[], Object[], Object...) invoke(MethodHandleInfo, null, null, Object...)}
	 *
	 */
	@Override
	protected Object invoke(MethodHandleInfo methodHandleInfo, Object... args)
	        throws InvocationException {

		try {
            return invoke(methodHandleInfo, null, null, args);
        } catch (NoSuchMethodException | SecurityException
                | InstantiationException | IllegalAccessException
                | IllegalArgumentException | InvocationTargetException e) {
            throw new InvocationException(e);
        }
	}

	/**
	 * {@inheritDoc}<p>
	 * Convenience method for instantiating objects with the default constructor when the underlying
	 * method is non-static.<p>
	 * Equivalent to calling {@link #invoke(MethodHandleInfo, Class[], Object[], ArgsFactory) invoke(MethodHandleInfo, null, null, ArgsFactory)}
	 *
	 */
	@Override
	protected Object invoke(MethodHandleInfo methodHandleInfo, ArgsFactory argsFactory)
	        throws ArgsProductionException, InvocationException {

		try {
            return invoke(methodHandleInfo, null, null, argsFactory);
        } catch (NoSuchMethodException | SecurityException
                | InstantiationException | IllegalAccessException
                | IllegalArgumentException | InvocationTargetException e) {
            throw new InvocationException(e);
        }
	}

	/**
	 * Invokes the methodhandle contained in the methodhandleinfo mapped to the provided key with
	 * the given arguments. <p> Attempts to instantiate an object with the provided constructor
	 * parameter types and arguments if the underlying method is non-static.
	 *
	 * @param key							the key to which the desired methodhandleinfo is mapped
	 * @param constructorParameterTypes		the parameter types of the desired constructor
	 * @param constructorArgs				the arguments to be passed to the desired constructor
	 * @param args							the arguments to be passed to the methodhandle
	 * @return								the result of invoking the methodhandle with the
	 * 										provided arguments
	 * @throws InvocationTargetException	See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws IllegalArgumentException		See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws IllegalAccessException		See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws InstantiationException		See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws SecurityException			See the documentation for
	 * 										{@link java.lang.Class#getConstructor(Class...)}.
	 * @throws NoSuchMethodException		See the documentation for
	 * 										{@link java.lang.Class#getConstructor(Class...)}.
	 * @throws UnmappedKeyException			If there is no methodhandleinfo associated with the
	 * 										provided key.
	 * @throws InvocationException          If methodhandle invocation fails.
	 *
	 */
	public Object invoke(K key, Class<?>[] constructorParameterTypes,
			Object[] constructorArgs, Object... args)
			throws NoSuchMethodException, SecurityException, InstantiationException,
			IllegalAccessException, IllegalArgumentException, InvocationTargetException,
			UnmappedKeyException, InvocationException {
		return invoke(getMethodHandleInfo(key), constructorParameterTypes, constructorArgs, args);
	}

	/**
	 * Invokes the methodhandle contained in the methodhandleinfo mapped to the provided key with
	 * the arguments produced by the given argsfactory. <p> Attempts to instantiate an object with
	 * the provided constructor parameter types and arguments if the underlying method is
	 * non-static.
	 *
	 * @param key							the key to which the desired methodhandleinfo is mapped
	 * @param constructorParameterTypes		the parameter types of the desired constructor
	 * @param constructorArgs				the arguments to be passed to the desired constructor
	 * @param argsFactory					Produces the arguments that are passed to the methodhandle.
	 * @return								the result of invoking the methodhandle with the produced
	 * 										arguments
	 * @throws InvocationTargetException	See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws IllegalArgumentException		See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws IllegalAccessException		See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws InstantiationException		See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws SecurityException			See the documentation for
	 * 										{@link java.lang.Class#getConstructor(Class...)}.
	 * @throws NoSuchMethodException		See the documentation for
	 * 										{@link java.lang.Class#getConstructor(Class...)}.
	 * @throws UnmappedKeyException			If there is no methodhandleinfo associated with the
	 * 										provided key.
	 * @throws ArgsProductionException		If the argsfactory fails to produce arguments.
	 * @throws InvocationException          If methodhandle invocation fails.
	 *
	 */
	public Object invoke(K key, Class<?>[] constructorParameterTypes,
			Object[] constructorArgs, ArgsFactory argsFactory)
			throws NoSuchMethodException, SecurityException, InstantiationException,
			IllegalAccessException, IllegalArgumentException, InvocationTargetException,
			ArgsProductionException, UnmappedKeyException, InvocationException {

		return invoke(getMethodHandleInfo(key), constructorParameterTypes, constructorArgs,
				argsFactory);
	}

	/**
	 * {@inheritDoc}<p>
	 * Convenience method for instantiating objects with the default constructor when the underlying
	 * method in non-static.<p>
	 * Equivalent to calling {@link #invoke(Object, Class[], Object[], Object...) invoke(Object, null, null, Object...)}
	 *
	 */
	@Override
	public Object invoke(K key, Object... args) throws UnmappedKeyException, InvocationException {
		try {
            return invoke(key, null, null, args);
        } catch (NoSuchMethodException | SecurityException
                | InstantiationException | IllegalAccessException
                | IllegalArgumentException | InvocationTargetException e) {
            throw new InvocationException(e);
        }
	}

	/**
	 * {@inheritDoc}<p>
	 * Convenience method for instantiating objects with the default constructor when the underlying
	 * method in non-static.<p>
	 * Equivalent to calling {@link #invoke(Object, Class[], Object[], ArgsFactory) invoke(Object, null, null, ArgsFactory)}
	 *
	 * @throws IllegalArgumentException		See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws SecurityException			See the documentation for
	 * 										{@link java.lang.Class#getConstructor(Class...)}.
	 *
	 */
	@Override
	public Object invoke(K key, ArgsFactory argsFactory)
            throws UnmappedKeyException, ArgsProductionException, InvocationException {
		try {
            return invoke(key, null, null, argsFactory);
        } catch (NoSuchMethodException | SecurityException
                | InstantiationException | IllegalAccessException
                | IllegalArgumentException | InvocationTargetException e) {
            throw new InvocationException(e);
        }
	}

	/**
	 * {@inheritDoc}<p>
	 * Uses the default constructor for non-static methods.
	 *
	 * @throws IllegalArgumentException		See the documentation for
	 * 										{@link java.lang.reflect.Constructor#newInstance(Object...)}.
	 * @throws SecurityException			See the documentation for
	 * 										{@link java.lang.Class#getConstructor(Class...)}.
	 *
	 */
	@Override
	public Object invoke(Object... args) throws UnmappedKeyException, InvocationException {
		KeyArgsPair<K> keyArgsPair;

		keyArgsPair = new KeyArgsPair<K>(args);

		try {
            return invoke(keyArgsPair.getKey(), null, null, keyArgsPair.getArgs());
        } catch (NoSuchMethodException | SecurityException
                | InstantiationException | IllegalAccessException
                | IllegalArgumentException | InvocationTargetException e) {
            throw new InvocationException(e);
        }
	}
}
