package io.magus.methodmap;

import io.magus.methodmap.error.ArgsProductionException;
import io.magus.methodmap.error.DuplicateKeyException;
import io.magus.methodmap.error.InvocationException;
import io.magus.methodmap.error.MapEntryKeyProductionException;
import io.magus.methodmap.error.UnmappedKeyException;

import java.lang.invoke.MethodHandles;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.HashMap;
import java.util.Objects;

/**
 * Base class for implementations of a MethodMap
 *
 * @author Enseart A. Simpson
 *
 * @param <K>	Type of the keys contained by the MethodMap
 */
public abstract class AbstractMethodMap<K> extends HashMap<K, MethodHandleInfo> {

	private static final long serialVersionUID = -3499174659234956205L;

	/**
	 * @param target							the target object to which all non-static
	 * 											methodhandles will be bound
	 * @param methods							the methods that will be included in the map
	 * @param entryKeyFactory					the factory used to generate keys for each of the
	 * 											provided methods
	 * @param lookup							the lookup object used to convert the provided
	 * 											methods to methodhandles
	 * @throws DuplicateKeyException			If more than one of the same key is generated for
	 * 											the entries of the methodmap.
	 * @throws IllegalAccessException			If MethodHandle lookup fails for any of the provided
	 * 											methods.
	 * @throws MapEntryKeyProductionException	If entry keys fail to be generated for any of the
	 * 											provided methods.
	 */
	protected AbstractMethodMap(Object target, Collection<Method> methods,
			MapEntryKeyFactory<K> entryKeyFactory, MethodHandles.Lookup lookup)
			throws DuplicateKeyException, IllegalAccessException, MapEntryKeyProductionException {
		super();

		K[] producedEntryKeys;
		MethodHandleInfo newMethodHandleInfo, mappedMethodHandleInfo;

		if(methods != null) {
			for(Method method : methods) {

				newMethodHandleInfo = new MethodHandleInfo(target, method, lookup);
				producedEntryKeys = entryKeyFactory.produceMapEntryKeys(newMethodHandleInfo);

				if(producedEntryKeys != null) {
					for(K producedEntryKey : producedEntryKeys) {
						mappedMethodHandleInfo = get(producedEntryKey);

						if(mappedMethodHandleInfo != null)
							throw new DuplicateKeyException(Objects.toString(producedEntryKey));

						put(producedEntryKey, newMethodHandleInfo);
					}
				}
			}
		}
	}

	/**
	 * Retrieves the methodhandleinfo mapped to the provided key.
	 *
	 * @param key					the key to which the desired methodhandleinfo is mapped
	 * @return						the methodhandleinfo mapped to the provided key
	 * @throws UnmappedKeyException	If there is no methodhandleinfo mapped to the provided key.
	 *
	 */
	public MethodHandleInfo getMethodHandleInfo(K key) throws UnmappedKeyException {
		MethodHandleInfo result;

		result = get(key);
		if(result == null)
			throw new UnmappedKeyException(Objects.toString(key));

		return result;
	}

	/**
	 * Invokes the methodhandle contained in the provided methodhandleinfo with the provided
	 * arguments.
	 *
	 * @param methodHandleInfo     the methodhandleinfo containing the methodhandle to be invoked
	 * @param args                 the arguments to be passed to the methodhandle
	 * @return                     the result of executing the methodhandle with the provided arguments
	 * @throws InvocationException If invocation fails.
	 */
	protected Object invoke(MethodHandleInfo methodHandleInfo, Object... args)
	        throws InvocationException {

	    try {
	        return methodHandleInfo.invoke(args);
	    } catch(Throwable e) {
	        throw new InvocationException(e);
	    }
	}

	/**
	 * Invokes the methodhandle contained in the provided methodhandleinfo with the arguments
	 * produced by the provided argsfactory.
	 *
	 * @param methodHandleInfo			the methodhandleinfo containing the methodhandle to
	 * 									be invoked
	 * @param argsFactory				Produces the arguments that are passed to the methodhandle.
	 * @return							the result of executing the methodhandle with the produced
	 * 									arguments
	 * @throws ArgsProductionException	If the argsfactory fails to produce arguments.
	 * @throws InvocationException		If invocation fails.
	 */
	protected Object invoke(MethodHandleInfo methodHandleInfo, ArgsFactory argsFactory)
			throws ArgsProductionException, InvocationException {
		return invoke(methodHandleInfo, argsFactory.produceArgs(methodHandleInfo));
	}

	/**
	 * Invokes the methodhandle contained in the methodhandleinfo mapped to the provided key with
	 * the given arguments.
	 *
	 * @param key					the key to which the desired methodhandleinfo is mapped
	 * @param args					the arguments to be passed to the methodhandle
	 * @return						the result of executing the methodhandle with the provided
	 * 								arguments
	 * @throws InvocationException	If invocation fails.
	 * @throws UnmappedKeyException	If there is no methodhandleinfo associated with the provided
	 * 								key.
	 */
	public abstract Object invoke(K key, Object... args)
	        throws UnmappedKeyException, InvocationException;

	/**
	 * Invokes the methodhandle contained in the methodhandleinfo mapped to the provided key with
	 * arguments produced by the given argsfactory.
	 *
	 * @param key						the key to which the desired methodhandleinfo is mapped
	 * @param argsFactory				Produces the arguments that are passed to the methodhandle.
	 * @return							the result of executing the methodhandle with the produced
	 * 									arguments
	 * @throws ArgsProductionException	If the argsfactory fails to produce arguments.
	 * @throws InvocationException		If invocation fails.
	 * @throws UnmappedKeyException		If there is no methodhandleinfo associated with the provided
	 * 									key.
	 *
	 */
	public abstract Object invoke(K key, ArgsFactory argsFactory)
			throws UnmappedKeyException, ArgsProductionException, InvocationException;

	/**
	 * Treats the first argument as the key, the remaining arguments as the methodhandle
	 * arguments, and invokes the methodhandle contained in the methodhandleinfo
	 * mapped to the provided key with the given arguments.
	 *
	 * @param args					the key to which the desired methodhandleinfo is mapped and the
	 * 								arguments to be passed to the methodhandle
	 * @return						the result of executing the methodhandle with the provided
	 * 								arguments
	 * @throws InvocationException	If invocation fails.
	 * @throws UnmappedKeyException	If there is no methodhandleinfo associated with the provided
	 * 								key.
	 */
	public abstract Object invoke(Object... args) throws UnmappedKeyException, InvocationException;
}
