package io.magus.methodmap;

import io.magus.methodmap.error.MethodCollectionProductionException;

import java.lang.reflect.Method;
import java.util.Collection;
import java.util.HashSet;

/**
 * Base class for implementations of a BoundMethodCollectionFactory. Instances of this class produce
 * method collections containing methods that belong strictly to the class of a provided object.
 * This factory is used mainly for the production of method collections for implementations of
 * {@link io.magus.methodmap.BoundMethodMap}.
 *
 * @author Enseart A. Simpson
 *
 */
public abstract class AbstractBoundMethodCollectionFactory implements MethodCollectionFactory {

	private final Object target;

	/**
	 * @param target	the target object whose class's methods are candidates to be included in the
	 * 					output of a call to {@link #produceMethodCollection()}
	 *
	 */
	protected AbstractBoundMethodCollectionFactory(Object target) {
		this.target = target;
	}

	@Override
	public final Collection<Method> produceMethodCollection()
			throws MethodCollectionProductionException {

		Collection<Method> result;
		Method[] methods;

		result = new HashSet<Method>();
		methods = target.getClass().getDeclaredMethods();

		for(Method method : methods)
			if(include(method))
				result.add(method);

		return result;
	}

	/**
	 * Retrieves the underlying object of this BoundMethodCollectionFactory.
	 *
	 * @return	the underlying object of this BoundMethodCollectionFactory
	 *
	 */
	public final Object getTarget() {
		return target;
	}

	/**
	 * Determines whether or not a candidate method should be included in the method collection
	 * produced by this BoundMethodCollectionFactory.
	 *
	 * @param method	method to be evaluated for inclusion in the output of
	 * 					{@link #produceMethodCollection()}
	 * @return			{@code true} if the method should be included in the produced
	 * 					method collection, {@code false} otherwise
	 */
	protected abstract boolean include(Method method);
}
