package io.loop.fusion.api;

import android.app.Service;
import android.content.ComponentName;
import android.content.pm.PackageManager;
import android.content.pm.ServiceInfo;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import java.util.ArrayList;
import java.util.List;

/**
 * A {@link Service} that will be used as the base for every module of Fusion. It contains the
 * base methods that should be defined, such as returning the ModuleApiVersion that is used
 * in the {@link FusionModule}.
 *
 * Currently, there are 3 types of {@link FusionModule}s :
 *
 * {@link FusionAction} - Provides an Action that should be performed by the client.
 * {@link FusionEvent} - Informs the core app that an Event happened in the client app.
 * {@link FusionState} - Informs the core app that a State is set in the client app.
 *
 * @author Alexandre Piveteau
 */
abstract class FusionModule<T> extends Service {

    /**
     * @return The {@link FusionSerializable<T>} that will be used to store the data. If no
     *          {@link FusionSerializable} is provided, you will not be able to save any data
     *          in the Fusion core app, so your {@link FusionAction} will not have any "parameters".
     */
    @Nullable
    public abstract FusionSerializable<T> getFusionSerializable();

    /**
     * @return The version number of the protocol. This allows the Fusion core app to play it "safe"
     *          if the protocol version does not seem compatible, and avoid any crashes if the
     *          module's API version is above the core app's. If you return a wrong version number,
     *          you are potentially facing multiple user crashes.
     */
    final int getModuleApiVersion(){
        return Fusion.Config.API_VERSION;
    }

    /**
     * @return The unique {@link String} identifier of the module. This will be used in the Fusion
     *          core application to store all the data associated with your module, so please be
     *          sure to have something unique within your package (the package name of your application
     *          is also used to store your module's data) and that will not change as long as you
     *          support this module, otherwise the module configuration of your user will be lost
     *          or will simply not work anymore.
     */
    @NonNull
    public abstract String getModuleIdentifier();

    /**
     * @return A {@link List} linking the possible {@link T} values to the possible {@link String}
     *          names that are available for this action.
     */
    @Nullable
    public abstract List<T> getPossibleValues();

    /**
     * @return A {@link List} linking the possible {@link byte[]} values for this action.
     *
     * @see {@link #getPossibleValues()}
     */
    /* package */ List<byte[]> getPossibleValuesByte() {

        // Return directly if appropriate.
        if(getPossibleValues() == null) return null;

        List<byte[]> list = new ArrayList<>();
        FusionSerializable<T> fusionSerializable = getFusionSerializable();

        if(fusionSerializable != null) {
            for(T item : getPossibleValues()) {
                list.add(fusionSerializable.serializeData(item));
            }
        }

        return list;
    }

    /**
     * @param t The {@link T} that we need to get the title from.
     * @return The human-readable {@link String} that describes this {@link T}.
     */
    @NonNull
    public abstract String getTitle(T t);

    /**
     * {@inheritDoc}
     */
    @Override
    public void onCreate() {
        super.onCreate();

        // The ComponentName that corresponds to the current FusionModule.
        ComponentName serviceComponentName = new ComponentName(this, getClass());
        try {
            ServiceInfo serviceInfo = getPackageManager().getServiceInfo(serviceComponentName, 0);

            // Verify if we have the correct permission set. If not, the FusionModule is not secure!
            if(!Fusion.Permissions.BIND_MODULE.equals(serviceInfo.permission)) {
                throw new IllegalStateException(String.format("%s is not declared with the permission \"%s\"", serviceComponentName.getShortClassName(), Fusion.Permissions.BIND_MODULE));
            }
        } catch (PackageManager.NameNotFoundException exception) {
            Log.e(FusionModule.class.getSimpleName(), String.format("Can't get ServiceInfo for %s", serviceComponentName.getShortClassName()));
        }
    }

    /**
     * Called whenever your {@link FusionAction} has been connected to the Fusion core. If you need
     * to instantiate anything specific for your {@link FusionAction}, this is when you should
     * do it.
     */
    public void onStartListening() {}

    /**
     * Called whenever your {@link FusionAction} has been disconnected from the Fusion core. If you
     * need to destroy anything specific of your {@link FusionAction}, this is when you should
     * do it.
     */
    public void onStopListening() {}
}
