package io.loop.fusion.api;

import android.content.Intent;
import android.os.IBinder;
import android.os.RemoteCallbackList;
import android.os.RemoteException;
import android.support.annotation.CallSuper;
import android.util.Log;

import java.util.List;

/**
 * An object representing a Fusion event. A {@link FusionEvent} is in charge of notifying
 * the core Fusion app of things that happen on the device, for instance that your app
 * has performed a certain action. It is similar in this point with a {@link FusionState}, although
 * a {@link FusionEvent} happens at really one point of the time.
 *
 * The implementation of the {@link FusionEvent} is based on the {@link IFusionEvent} and
 * the {@link IFusionEventCallback} AIDL definitions. If you want to implement your custom own service
 * that communicates directly with the Fusion application, give a look at the AIDL definition and
 * implement it in your service.
 *
 * @param <T> The type of the data that will be handled by the {@link FusionEvent}. It can
 *              be really anything, as long as you provide the appropriate {@link FusionSerializable}.
 *
 * @author Alexandre Piveteau
 */
public abstract class FusionEvent<T> extends FusionModule<T> implements FusionMatcher<T> {

    /**
     * The {@link IBinder} that is used specifically for a {@link FusionEvent}.
     */
    private IBinder mBinder = new IFusionEvent.Stub() {

        /**
         * {@inheritDoc}
         */
        @Override
        public int getModuleApiVersion() throws RemoteException {
            // Call to the containing class.
            return FusionEvent.this.getModuleApiVersion();
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getModuleIdentifier() throws RemoteException {
            // Call to the containing class.
            return FusionEvent.this.getModuleIdentifier();
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public List<byte[]> getPossibleValues() throws RemoteException {
            // Call to the containing class.
            return FusionEvent.this.getPossibleValuesByte();
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getTitle(byte[] data) throws RemoteException {
            // Call to the containing class.
            return FusionEvent.this.getTitle(data);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean match(byte[] event, byte[] savedEvent) throws RemoteException {
            // Call to the containing class.
            return FusionEvent.this.match(event, savedEvent);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public void registerEventCallback(IFusionEventCallback callback) throws RemoteException {
            // Call to the containing class.
            FusionEvent.this.registerEventCallback(callback);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public void unregisterEventCallback(IFusionEventCallback callback) throws RemoteException {
            // Call to the containing class.
            FusionEvent.this.unregisterEventCallback(callback);
        }
    };

    /**
     * A {@link RemoteCallbackList} containing all the callbacks that are registered to this
     * {@link FusionEvent}.
     */
    private RemoteCallbackList<IFusionEventCallback> mRemoteCallbacks;

    /**
     * @see {@link #mBinder#getTitle(byte[])}
     */
    private String getTitle(byte[] bytes) {

        T data = null;
        FusionSerializable<T> fusionSerializable = getFusionSerializable();

        // Deserialize the data if it is actually possible to do so.
        if(fusionSerializable != null) {
            data = fusionSerializable.deserializeData(bytes);
        }

        // Call the callback.
        return getTitle(data);
    }

    /**
     * @see {@link #mBinder#match(byte[], byte[])}
     */
    private boolean match(byte[] bytes, byte[] savedBytes) {

        T data = null, savedData = null;
        FusionSerializable<T> fusionSerializable = getFusionSerializable();

        // Deserialize the data if it is actually possible to do so.
        if(fusionSerializable != null) {
            data = fusionSerializable.deserializeData(bytes);
            savedData = fusionSerializable.deserializeData(savedBytes);
        }

        // Call the callback.
        return match(data, savedData);
    }

    /**
     * Performs the {@link FusionEvent}. You should call this each time that the event has been
     * performed and that you would like to notify the core Fusion app about it.
     *
     * @param data The {@link T} associated with the event.
     */
    public final void notifyEvent(T data) {

        FusionSerializable<T> serializable = getFusionSerializable();
        byte[] serializedData = null;

        if(serializable != null) {
            serializedData = serializable.serializeData(data);
        }

        int callbackCount = mRemoteCallbacks.beginBroadcast();

        for(int i = 0; i < callbackCount; i++) {
            IFusionEventCallback callback = mRemoteCallbacks.getBroadcastItem(i);

            try {
                callback.performEvent(serializedData);
            } catch (RemoteException remoteException) {
                // Ignore. The callback was not available, and the system will remove it for us.
                Log.d(FusionEvent.class.getSimpleName(), "Could not send performEvent through IPC.");
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    @CallSuper
    @Override
    public IBinder onBind(Intent intent) {

        // Return the appropriate IFusionAction Binder.
        if(intent.getAction().equals(Fusion.Intents.EVENT)) {

            // Call the appropriate callbacks.
            onStartListening();

            return mBinder;
        }

        // We have no appropriate Binder for this Intent.
        return null;
    }

    /**
     * {@inheritDoc}
     */
    @CallSuper
    @Override
    public void onCreate() {
        super.onCreate();

        // Instantiate our RemoteCallbackList.
        mRemoteCallbacks = new RemoteCallbackList<>();
    }

    /**
     * {@inheritDoc}
     */
    @CallSuper
    @Override
    public void onDestroy() {

        // The Service is killed, so we should remove all the callbacks.
        mRemoteCallbacks.kill();

        super.onDestroy();
    }

    /**
     * {@inheritDoc}
     */
    @CallSuper
    @Override
    public boolean onUnbind(Intent intent) {

        // Call the appropriate callbacks.
        if(intent.getAction().equals(Fusion.Intents.EVENT)) {
            this.onStopListening();
        }

        return super.onUnbind(intent);
    }

    /**
     * @see {@link #mBinder#registerEventCallback(IFusionEventCallback)}
     */
    private void registerEventCallback(IFusionEventCallback callback) throws RemoteException {
        mRemoteCallbacks.register(callback);
    }

    /**
     * @see {@link #mBinder#unregisterEventCallback(IFusionEventCallback)}
     */
    private void unregisterEventCallback(IFusionEventCallback callback) throws RemoteException {
        mRemoteCallbacks.unregister(callback);
    }
}
