package io.loop.fusion.api;

import android.content.Intent;
import android.os.IBinder;
import android.os.RemoteException;
import android.support.annotation.CallSuper;
import android.support.annotation.Nullable;

import java.util.List;

/**
 * An object representing a Fusion action. A {@link FusionAction} is in charge of making things
 * happen on the user's device, each time it is triggered by the fusion app.
 *
 * This implementation of the {@link FusionAction} is based on the {@link IFusionAction} AIDL
 * definition. If you want to implement your custom own service that communicates directly
 * with the Fusion application, give a look at the AIDL definition and implement
 * it in your service.
 *
 * @param <T> The type of the data that will be handled by the {@link FusionAction}. It can
 *              be really anything, as long as you provide the appropriate {@link FusionSerializable}.
 *
 * @author Alexandre Piveteau
 */
public abstract class FusionAction<T> extends FusionModule<T> {

    /**
     * The {@link IBinder} that is used specifically for a {@link FusionAction}.
     */
    private IBinder mBinder = new IFusionAction.Stub() {

        /**
         * {@inheritDoc}
         */
        @Override
        public int getModuleApiVersion() throws RemoteException {
            // Call to the containing class.
            return FusionAction.this.getModuleApiVersion();
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getModuleIdentifier() throws RemoteException {
            // Call to the containing class.
            return FusionAction.this.getModuleIdentifier();
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public void performAction(byte[] data) throws RemoteException {
            // Call to the containing class.
            FusionAction.this.performAction(data);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public List<byte[]> getPossibleValues() throws RemoteException {
            // Call to the containing class.
            return FusionAction.this.getPossibleValuesByte();
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getTitle(byte[] data) throws RemoteException {
            return FusionAction.this.getTitle(data);
        }
    };

    /**
     * @see {@link #mBinder#getTitle(byte[])}
     */
    private String getTitle(byte[] bytes) {

        T data = null;
        FusionSerializable<T> fusionSerializable = getFusionSerializable();

        // Deserialize the data if it is actually possible to do so.
        if(fusionSerializable != null) {
            data = fusionSerializable.deserializeData(bytes);
        }

        // Call the callback.
        return getTitle(data);
    }

    /**
     * {@inheritDoc}
     */
    @CallSuper
    @Override
    public IBinder onBind(Intent intent) {

        // Return the appropriate IFusionAction Binder.
        if(intent.getAction().equals(Fusion.Intents.ACTION)) {

            // Call the appropriate callbacks.
            onStartListening();

            return mBinder;
        }

        // We have no appropriate Binder for this Intent.
        return null;
    }

    /**
     * Called whenever the Fusion core thinks you should actually execute your {@link FusionAction}.
     *
     * @param data The {@link T} parameters that you should use for this {@link FusionAction}.
     */
    public abstract void onPerformAction(@Nullable T data);

    /**
     * @see {@link #mBinder#performAction(byte[])}
     */
    private void performAction(byte[] bytes) throws RemoteException {

        T data = null;
        FusionSerializable<T> fusionSerializable = getFusionSerializable();

        // Deserialize the data if it is actually possible to do so.
        if(fusionSerializable != null) {
            data = fusionSerializable.deserializeData(bytes);
        }

        // Call the callback.
        onPerformAction(data);
    }

    /**
     * {@inheritDoc}
     */
    @CallSuper
    @Override
    public boolean onUnbind(Intent intent) {

        // Call the appropriate callbacks.
        if(intent.getAction().equals(Fusion.Intents.ACTION)) {
            this.onStopListening();
        }

        return super.onUnbind(intent);
    }
}
