package io.loop.fusion.api;

import android.content.Intent;
import android.os.IBinder;
import android.os.RemoteCallbackList;
import android.os.RemoteException;
import android.support.annotation.CallSuper;
import android.util.Log;

import java.util.List;

/**
 * An object representing a Fusion state. A {@link FusionState} is in charge of representing
 * the state of a component in an application. Contrary to the {@link FusionEvent}, a
 * {@link FusionState} is set during a period of time; it is not a simple event happening on the
 * device.
 *
 * The implementation of the {@link FusionState} is based on the {@link IFusionState} and
 * the {@link IFusionStateCallback} AIDL definitions. If you want to implement your own service
 * that communicates directly with the Fusion application, give a look at the AIDL definition
 * and implement it in your service
 *
 * @param <T> The type of the data that will be handled by the {@link FusionState}. It can
 *              be really anything, as long as you provide the appropriate {@link FusionSerializable}.
 *
 * @author Alexandre Piveteau
 */
public abstract class FusionState<T> extends FusionModule<T> implements FusionMatcher<T> {

    /**
     * The {@link IBinder} that is used specifically for a {@link FusionState}.
     */
    private IBinder mBinder = new IFusionState.Stub() {

        /**
         * {@inheritDoc}
         */
        @Override
        public int getModuleApiVersion() throws RemoteException {
            // Call to the containing class.
            return FusionState.this.getModuleApiVersion();
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getModuleIdentifier() throws RemoteException {
            // Call to the containing class.
            return FusionState.this.getModuleIdentifier();
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public List getPossibleValues() throws RemoteException {
            // Call to the containing class.
            return FusionState.this.getPossibleValues();
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public byte[] getState() throws RemoteException {
            // Call to the containing class.
            return FusionState.this.getStateData();
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getTitle(byte[] data) throws RemoteException {
            // Call to the containing class.
            return FusionState.this.getTitle(data);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean match(byte[] state, byte[] savedState) throws RemoteException {
            // Call to the containing class.
            return FusionState.this.match(state, savedState);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public void registerStateCallback(IFusionStateCallback callback) throws RemoteException {
            // Call to the containing class.
            FusionState.this.registerStateCallback(callback);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public void unregisterStateCallback(IFusionStateCallback callback) throws RemoteException {
            // Call to the containing class.
            FusionState.this.unregisterStateCallback(callback);
        }
    };

    /**
     * A {@link RemoteCallbackList} containing all the callbacks that are registered to this
     * {@link FusionState}.
     */
    private RemoteCallbackList<IFusionStateCallback> mRemoteCallbacks;

    /**
     * @see {@link #mBinder#getTitle(byte[])}
     */
    private String getTitle(byte[] bytes) {

        T data = null;
        FusionSerializable<T> fusionSerializable = getFusionSerializable();

        // Deserialize the data if it is actually possible to do so.
        if(fusionSerializable != null) {
            data = fusionSerializable.deserializeData(bytes);
        }

        // Call the callback.
        return getTitle(data);
    }

    /**
     * Method called by the Fusion core to know which state is active in this {@link FusionState}.
     *
     * @return The {@link T} corresponding to the state.
     */
    public abstract T getState();

    /**
     * @return The serialized {@link T} of the current state.
     */
    private byte[] getStateData() {

        FusionSerializable<T> serializable = getFusionSerializable();

        if(serializable != null) {
            return serializable.serializeData(getState());
        } else {
            return null;
        }
    }

    /**
     * @see {@link #mBinder#match(byte[], byte[])}
     */
    private boolean match(byte[] bytes, byte[] savedBytes) {

        T data = null, savedData = null;
        FusionSerializable<T> fusionSerializable = getFusionSerializable();

        // Deserialize the data if it is actually possible to do so.
        if(fusionSerializable != null) {
            data = fusionSerializable.deserializeData(bytes);
            savedData = fusionSerializable.deserializeData(savedBytes);
        }

        // Call the callback.
        return match(data, savedData);
    }

    /**
     * Notifies the Fusion core app that the state has changed in this {@link FusionState}, and that
     * the newest state {@link T} can be queried from the {@link #getState()} method.
     *
     * It is recommended to call this method as less as possible; since it requires some processing
     * from the core app, this could decrease the battery life of users and even slow down
     * lower-end devices !
     */
    public final void notifyStateChanged() {

        int callbackCount = mRemoteCallbacks.beginBroadcast();

        for(int i = 0; i < callbackCount; i++) {
            IFusionStateCallback callback = mRemoteCallbacks.getBroadcastItem(i);

            try {
                callback.performStateChange();
            } catch (RemoteException exception) {
                // Ignore. The callback was not available, and the system will remove it for us.
                Log.d(FusionState.class.getSimpleName(), "Could not send performStateChanged through IPC.");
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    @CallSuper
    @Override
    public IBinder onBind(Intent intent) {

        // Return the appropriate IFusionAction Binder.
        if(intent.getAction().equals(Fusion.Intents.STATE)) {

            // Call the appropriate callbacks.
            onStartListening();

            return mBinder;
        }

        // We have no appropriate Binder for this Intent.
        return null;
    }

    /**
     * {@inheritDoc}
     */
    @CallSuper
    @Override
    public void onCreate() {
        super.onCreate();

        // Instantiate our RemoteCallbackList.
        mRemoteCallbacks = new RemoteCallbackList<>();
    }

    /**
     * {@inheritDoc}
     */
    @CallSuper
    @Override
    public void onDestroy() {

        // The Service is killed, so we should remove all the callbacks.
        mRemoteCallbacks.kill();

        super.onDestroy();
    }

    /**
     * {@inheritDoc}
     */
    @CallSuper
    @Override
    public boolean onUnbind(Intent intent) {

        // Call the appropriate callbacks.
        if(intent.getAction().equals(Fusion.Intents.STATE)) {
            this.onStopListening();
        }

        return super.onUnbind(intent);
    }

    /**
     * @see {@link #mBinder#registerStateCallback(IFusionStateCallback)}
     */
    private void registerStateCallback(IFusionStateCallback callback) throws RemoteException {
        mRemoteCallbacks.register(callback);
    }

    /**
     * @see {@link #mBinder#unregisterStateCallback(IFusionStateCallback)}
     */
    private void unregisterStateCallback(IFusionStateCallback callback) throws RemoteException {
        mRemoteCallbacks.unregister(callback);
    }
}
