package io.loop.fusion.api;

import android.support.annotation.StringDef;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

/**
 * An interface defining the behavior of serialized data with that will be passed to {@link Fusion}.
 * This interface is already implemented for some specific data types that you might want to
 * use in {@link Fusion}, because they are supposed to be working symmetrically with the core
 * app too.
 *
 * A {@link FusionSerializable} {@link T} can be used as a parameter by the user when he
 * wants to create his own scripts. Please be sure that you use the built-in serializers if your
 * app uses this kind of values. Otherwise, your data will probably not be used by the user as
 * it should have been.
 *
 * @see {@link Boolean}
 * @see {@link String}
 *
 * @author Alexandre Piveteau
 */
public interface FusionSerializable<T> {

    /**
     * A {@link FusionSerializable< java.lang.Boolean >}. This implementation is included in the core app, so you
     * should use it each time you need a {@link java.lang.Boolean} as a parameter.
     */
    class Boolean implements FusionSerializable<java.lang.Boolean> {

        /**
         * A {@link FusionSerializable< java.lang.String >} that we will use as a helper for String serialization.
         */
        private FusionSerializable<java.lang.String> mStringFusionSerializable = new String();

        /**
         * {@inheritDoc}
         */
        @Override
        public java.lang.Boolean deserializeData(byte[] bytes) {
            return java.lang.Boolean.parseBoolean(mStringFusionSerializable.deserializeData(bytes));
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public java.lang.String getType() {
            return Type.TYPE_BOOLEAN;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public byte[] serializeData(java.lang.Boolean data) {
            return mStringFusionSerializable.serializeData(java.lang.Boolean.toString(data));
        }
    }

    /**
     * A {@link FusionSerializable< java.lang.Double >}. This implementation is included in the core app, so you
     * should use it each time you need a Double as a parameter.
     */
    class Double implements FusionSerializable<java.lang.Double> {

        /**
         * A {@link FusionSerializable< java.lang.String >} that we will use as a helper for String serialization.
         */
        private FusionSerializable<java.lang.String> mStringFusionSerializable = new String();

        /**
         * {@inheritDoc}
         */
        @Override
        public java.lang.Double deserializeData(byte[] bytes) {
            return java.lang.Double.parseDouble(mStringFusionSerializable.deserializeData(bytes));
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public java.lang.String getType() {
            return Type.TYPE_DOUBLE;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public byte[] serializeData(java.lang.Double data) {
            return mStringFusionSerializable.serializeData(java.lang.Double.toString(data));
        }
    }

    /**
     * A {@link FusionSerializable< java.lang.Integer >}. This implementation is included in the core app, so you
     * should use it each time you need a {@link java.lang.Integer} as a parameter.
     */
    class Integer implements FusionSerializable<java.lang.Integer> {

        /**
         * A {@link FusionSerializable< java.lang.String >} that we will use as a helper for String serialization.
         */
        private FusionSerializable<java.lang.String> mStringFusionSerializable = new String();

        /**
         * {@inheritDoc}
         */
        @Override
        public java.lang.Integer deserializeData(byte[] bytes) {
            return java.lang.Integer.parseInt(mStringFusionSerializable.deserializeData(bytes));
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public java.lang.String getType() {
            return Type.TYPE_INTEGER;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public byte[] serializeData(java.lang.Integer data) {
            return mStringFusionSerializable.serializeData(java.lang.Integer.toString(data));
        }
    }

    /**
     * A {@link FusionSerializable< java.lang.Long >}. This implementation is included in the core app, so you
     * should use it each time you need a {@link java.lang.Long} as a parameter.
     */
    class Long implements FusionSerializable<java.lang.Long> {

        /**
         * A {@link FusionSerializable< java.lang.String >} that we will use as a helper for String serialization.
         */
        private FusionSerializable<java.lang.String> mStringFusionSerializable = new String();

        /**
         * {@inheritDoc}
         */
        @Override
        public java.lang.Long deserializeData(byte[] bytes) {
            return java.lang.Long.parseLong(mStringFusionSerializable.deserializeData(bytes));
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public java.lang.String getType() {
            return Type.TYPE_LONG;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public byte[] serializeData(java.lang.Long data) {
            return mStringFusionSerializable.serializeData(java.lang.Long.toString(data));
        }
    }

    /**
     * A {@link FusionSerializable< java.lang.String >}. This implementation is included in the core app, so you
     * should use it each time you need a String as a parameter.
     */
    class String implements FusionSerializable<java.lang.String> {

        /**
         * {@inheritDoc}
         */
        @Override
        public java.lang.String deserializeData(byte[] bytes) {
            return new java.lang.String(bytes);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public java.lang.String getType() {
            return Type.TYPE_STRING;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public byte[] serializeData(java.lang.String data) {
            return data.getBytes();
        }
    }

    /**
     * An annotation defining the different types of {@link FusionSerializable} that are provided
     * and supported by the Fusion API.
     *
     * You should not return anything else than {@link Type#TYPE_UNDEFINED} if you are building your
     * own {@link FusionSerializable}, since it is very probably not supported by the core app yet.
     */
    @Retention(RetentionPolicy.SOURCE)
    @StringDef({
            Type.TYPE_BOOLEAN,
            Type.TYPE_DOUBLE,
            Type.TYPE_INTEGER,
            Type.TYPE_LONG,
            Type.TYPE_STRING,
            Type.TYPE_UNDEFINED
    })
    @interface Type {

        /**
         * A {@link Type} defining the {@link FusionSerializable} is a {@link java.lang.Boolean}.
         */
        java.lang.String TYPE_BOOLEAN = "java.lang.Boolean";

        /**
         * A {@link Type} defining the {@link FusionSerializable} is a {@link java.lang.Double}.
         */
        java.lang.String TYPE_DOUBLE = "java.lang.Double";

        /**
         * A {@link Type} defining the {@link FusionSerializable} is an {@link java.lang.Integer}.
         */
        java.lang.String TYPE_INTEGER = "java.lang.Integer";

        /**
         * A {@link Type} defining the {@link FusionSerializable} is a {@link java.lang.Long}.
         */
        java.lang.String TYPE_LONG = "java.lang.Long";

        /**
         * A {@link Type} defining that the {@link FusionSerializable} is a {@link java.lang.String}.
         */
        java.lang.String TYPE_STRING = "java.lang.String";

        /**
         * A {@link Type} defining that the {@link FusionSerializable} is unknown. <b>This should
         * be used by default for your custom {@link FusionSerializable} types.</b>
         */
        java.lang.String TYPE_UNDEFINED = "undefined";
    }

    /**
     * A method that is used to deserialize some byte[] to a {@link T} object. Since we are
     * using {@link T} objects in the rest of the {@link FusionAction} API, we want to ensure
     * that these items are built correctly when coming from the Fusion app.
     *
     * @param bytes The byte[] containing the object data.
     * @return The {@link T} object that is built.
     */
    T deserializeData(byte[] bytes);

    /**
     * @return The {@link Type} that is supported by this {@link FusionSerializable}. You should
     * <b>never</b> return anything else than {@link Type#TYPE_UNDEFINED} if you are writing your
     * own {@link FusionSerializable} class, since it is probably not supported by the system.
     */
    @Type
    java.lang.String getType();

    /**
     * A method that is used to serialize a {@link T} object to a byte[]. Since we are
     * using {@link T} objects in the rest of the {@link FusionAction} API, we want to ensure
     * that these items are saved correctly when going to the Fusion app.
     *
     * @param data The {@link T} object.
     * @return The byte[] corresponding to the {@link T}.
     */
    byte[] serializeData(T data);
}
