package io.loop.fusion.api;

/**
 * A class containing the general constants that should be used by the modules of Fusion.
 * Since the Fusion app uses these constants too, this allows you to ensure that you will
 * send and receive the appropriate events (in a way that is safe for your data, too).
 *
 * Moreover, this allows you to create your own components that will communicate with the
 * Fusion service instead of using the API library ones.
 *
 * <br>Usually, you do not need to interact directly with the constants in this file, although
 * you might be interested if you feel up to create your own module API for interacting
 * with Fusion.</br>
 *
 * @author Alexandre Piveteau
 */
final class Fusion {

    // Prevent instance creation.
    private Fusion(){}

    /**
     * A class containing all of the constants defining the current configuration state of
     * the library. It contains items such as the version number, for easily accessing
     * them through the API.
     */
    static final class Config {

        // Prevent instance creation.
        private Config(){}

        /**
         * The version code defining the current Fusion API. It is often required to return this,
         * so the Fusion app will not try to perform some work that should actually not be
         * performed.
         */
        public static final int API_VERSION = 2;
    }

    /**
     * A class containing all the {@link android.content.Intent} actions that you might need
     * to use in your application. These {@link android.content.Intent} actions are used by the
     * framework internally to bind to your actions, states and events.
     */
    static final class Intents {

        // Prevent instance creation.
        private Intents(){}

        /**
         * The string value of the <br>Action</br> module intent.
         */
        public static final String ACTION = "io.loop.fusion.module.ACTION";

        /**
         * The string value of the <br>Event</br> module intent.
         */
        public static final String EVENT = "io.loop.fusion.module.EVENT";

        /**
         * The string value of the <br>State</br> module intent.
         */
        public static final String STATE = "io.loop.fusion.module.STATE";
    }

    /**
     * A class containing all the permissions that might be used in {@link FusionModule}s.
     * The content of these permissions is common to what you find in the Fusion core application,
     * so you should not change them.
     */
    static final class Permissions {

        // Prevent instance creation.
        private Permissions(){}

        /**
         * A permission that is required to be bound to a {@link FusionModule}. This ensures that
         * only selected apps will be able to connect to the modules and reduces the security
         * risks.
         */
        public static final String BIND_MODULE = "io.loop.fusion.permission.BIND_MODULE";
    }
}
