package org.springframework.boot.autoconfigure.web;

import java.lang.reflect.Field;
import java.util.Map;
import java.util.Properties;

import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.ReflectionUtils;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;
import org.springframework.web.servlet.mvc.WebContentInterceptor;

@Configuration
@ConditionalOnWebApplication
@ConditionalOnClass(WebMvcConfigurerAdapter.class)
@ConditionalOnMissingBean({ WebContentInterceptor.class })
@ConditionalOnProperty(prefix = "spring.mvc.cache", name = "enabled", havingValue = "true", matchIfMissing = true)
public class WebContentAutoConfiguration {

  @Configuration
  protected static class WebContentConfiguration extends WebMvcConfigurerAdapter {
    @Bean
    @ConfigurationProperties("spring.mvc.cache")
    public WebContentInterceptor webContentInterceptor() {
      WebContentInterceptor webContentInterceptor = new WebContentInterceptor() {
        @SuppressWarnings("unused")
        public void setCacheMappings(Map<String, Integer> cacheMappings) {
          Properties properties = new Properties();
          properties.putAll(cacheMappings);
          super.setCacheMappings(properties);
        }

        @SuppressWarnings("unused")
        public Object getCacheMappings() {
          Field field = ReflectionUtils.findField(this.getClass(), "cacheMappings");
          ReflectionUtils.makeAccessible(field);
          return ReflectionUtils.getField(field, this);
        }
      };
      return webContentInterceptor;
    }

    @Override
    public void addInterceptors(InterceptorRegistry registry) {
      registry.addInterceptor(webContentInterceptor());
    }
  }
}
