package org.springframework.boot.autoconfigure.web;

import java.util.Arrays;
import java.util.LinkedHashSet;

import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.web.WebClientAutoConfiguration.RestTemplateConfiguration;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.client.RestTemplateCustomizer;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.HttpStatus;
import org.springframework.http.client.BufferingClientHttpRequestFactory;
import org.springframework.http.client.ClientHttpRequestFactoryCustom;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.LoggingClientHttpRequestInterceptor;
import org.springframework.http.client.ResponseErrorHandlerCustom;
import org.springframework.web.client.RestTemplate;

/**
 * @see org.springframework.util.SocketUtils
 */
@Configuration
@AutoConfigureAfter(WebClientAutoConfiguration.class)
public class WebClientAutoConfigurationAfter {
  @Configuration
  @ConditionalOnClass(RestTemplate.class)
  @AutoConfigureBefore(RestTemplateConfiguration.class)
  @EnableConfigurationProperties(HttpClientProperties.class)
  public static class RestTemplateConfigurationCustom {
    private final HttpClientProperties clientProperties;

    public RestTemplateConfigurationCustom(HttpClientProperties clientProperties) {
      this.clientProperties = clientProperties;
    }

    @Bean
    public RestTemplateCustomizer clientHttpRequestFactory() {
      return new RestTemplateCustomizer() {
        @Override
        public void customize(RestTemplate restTemplate) {
          ClientHttpRequestFactoryCustom clientHttpRequestFactory = new ClientHttpRequestFactoryCustom();
          clientHttpRequestFactory.setProxy(clientProperties.createProxy());
          // clientHttpRequestFactory.setOutputStreaming(false);
          restTemplate.setRequestFactory(new BufferingClientHttpRequestFactory(clientHttpRequestFactory));
        }
      };
    }

    @Bean
    public RestTemplateCustomizer clientHttpRequestInterceptor() {
      return new RestTemplateCustomizer() {
        @Override
        public void customize(RestTemplate restTemplate) {
          LoggingClientHttpRequestInterceptor interceptor = new LoggingClientHttpRequestInterceptor();
          interceptor.setStatus(HttpStatus.NOT_FOUND);

          restTemplate.getInterceptors().addAll(new LinkedHashSet<ClientHttpRequestInterceptor>(Arrays.asList(interceptor)));
        }
      };
    }

    @Bean
    public RestTemplateCustomizer responseErrorHandler() {
      return new RestTemplateCustomizer() {
        @Override
        public void customize(RestTemplate restTemplate) {
          ResponseErrorHandlerCustom errorHandler = new ResponseErrorHandlerCustom();
          errorHandler.setStatus(HttpStatus.BAD_REQUEST);
          restTemplate.setErrorHandler(errorHandler);
        }
      };
    }
  }

}
