package org.springframework.boot.autoconfigure.web;

import java.util.ArrayList;
import java.util.List;
import java.util.Map.Entry;

import javax.servlet.MultipartConfigElement;
import javax.servlet.Servlet;
import javax.servlet.http.HttpServletRequest;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.web.filter.OrderedMultipartFilter;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.ExtensionUnsupportedException;
import org.springframework.web.multipart.MultipartException;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;
import org.springframework.web.multipart.MultipartResolver;
import org.springframework.web.multipart.support.StandardServletMultipartResolver;
import org.springframework.web.servlet.DispatcherServlet;

@Configuration
@ConditionalOnClass({ Servlet.class, StandardServletMultipartResolver.class, MultipartConfigElement.class })
@ConditionalOnProperty(prefix = "spring.http.multipart", name = "enabled", matchIfMissing = true)
@AutoConfigureAfter({ MultipartAutoConfiguration.class })
public class MultipartAutoConfigurationAfter {

  @Primary
  @ConfigurationProperties("spring.multipart")
  @Bean(name = DispatcherServlet.MULTIPART_RESOLVER_BEAN_NAME)
  public MultipartResolver multipartResolver(MultipartProperties multipartProperties) {
    StandardServletMultipartResolver multipartResolver = new StandardServletMultipartResolver() {
      private List<String> allows = new ArrayList<String>();
      private List<String> denys = new ArrayList<String>();

      @Override
      public boolean isMultipart(HttpServletRequest request) {
        if (!"post".equalsIgnoreCase(request.getMethod()) && !"put".equalsIgnoreCase(request.getMethod())) {
          return false;
        }
        String contentType = request.getContentType();
        return (contentType != null && contentType.toLowerCase().startsWith("multipart/"));
      }

      @Override
      public MultipartHttpServletRequest resolveMultipart(HttpServletRequest request) throws MultipartException {
        MultipartHttpServletRequest multipartHttpServletRequest = super.resolveMultipart(request);
        for (Entry<String, List<MultipartFile>> entry : multipartHttpServletRequest.getMultiFileMap().entrySet()) {
          for (MultipartFile multipartFile : entry.getValue()) {
            String extension = StringUtils.getFilenameExtension(multipartFile.getOriginalFilename());
            if (StringUtils.hasText(extension) && !CollectionUtils.isEmpty(this.allows) && !this.allows.contains(extension.toLowerCase())) {
              throw new ExtensionUnsupportedException(extension);
            }
            if (StringUtils.hasText(extension) && !CollectionUtils.isEmpty(this.denys) && this.denys.contains(extension.toLowerCase())) {
              throw new ExtensionUnsupportedException(extension);
            }
          }
        }
        return multipartHttpServletRequest;
      }

      @SuppressWarnings("unused")
      public List<String> getAllows() {
        return allows;
      }

      @SuppressWarnings("unused")
      public void setAllows(List<String> allows) {
        this.allows = allows;
      }

      @SuppressWarnings("unused")
      public List<String> getDenys() {
        return denys;
      }

      @SuppressWarnings("unused")
      public void setDenys(List<String> denys) {
        this.denys = denys;
      }
    };
    multipartResolver.setResolveLazily(multipartProperties.isResolveLazily());
    return multipartResolver;
  }

  @Bean
  @ConfigurationProperties("spring.multipart.filter")
  @ConditionalOnProperty(prefix = "spring.multipart.filter", name = "enabled", matchIfMissing = true)
  public OrderedMultipartFilter multipartFilter(@Qualifier(DispatcherServlet.MULTIPART_RESOLVER_BEAN_NAME) MultipartResolver multipartResolver) {
    // String[] beanNames =
    // applicationContext.getBeanNamesForType(MultipartResolver.class);
    // Assert.notEmpty(beanNames, "No MultipartResolver found: not in a DispatcherServlet
    // request?");
    // Assert.isTrue(beanNames.length < 2, "Multiple MultipartResolver found: in a
    // DispatcherServlet request? " + StringUtils.arrayToCommaDelimitedString(beanNames));

    OrderedMultipartFilter multipartFilter = new OrderedMultipartFilter(multipartResolver);
    // multipartFilter.setMultipartResolverBeanName(beanNames[0]);
    return multipartFilter;
  }

}
