package org.springframework.boot.autoconfigure.web;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.jsoup.Jsoup;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.http.converter.BufferedImageHttpMessageConverterCustom;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.InputStreamHttpMessageConverter;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.http.converter.StringHttpMessageConverterCustom;
import org.springframework.http.converter.json.AbstractJackson2HttpMessageConverter;
import org.springframework.http.converter.json.GsonHttpMessageConverter;
import org.springframework.http.converter.xml.AbstractXmlHttpMessageConverter;
import org.springframework.http.converter.xml.Jaxb2RootElementHttpMessageConverter;
import org.springframework.http.converter.xml.MappingJackson2XmlHttpMessageConverter;

@Configuration
@ConditionalOnClass(HttpMessageConverter.class)
@AutoConfigureAfter({ HttpMessageConvertersAutoConfiguration.class })
public class HttpMessageConvertersAutoConfigurationAfter {

  private final List<HttpMessageConverter<?>> converters;

  public HttpMessageConvertersAutoConfigurationAfter(ObjectProvider<List<HttpMessageConverter<?>>> convertersProvider) {
    this.converters = convertersProvider.getIfAvailable() == null ? Collections.<HttpMessageConverter<?>>emptyList() : convertersProvider.getIfAvailable();
  }

  /**
   * @see org.springframework.core.annotation.AnnotationAwareOrderComparator#sort(java.util.List)
   * @see javax.xml.bind.Marshaller#setProperty(String, Object)
   * @see javax.xml.bind.Marshaller#JAXB_FORMATTED_OUTPUT
   */
  @Bean
  public HttpMessageConverters messageConverters() {
    List<HttpMessageConverter<?>> converters = new ArrayList<HttpMessageConverter<?>>();

    for (Iterator<HttpMessageConverter<?>> iterator = new HttpMessageConverters().getConverters().iterator(); iterator.hasNext();) {
      HttpMessageConverter<?> next = iterator.next();
      if (next instanceof StringHttpMessageConverter || next instanceof AbstractXmlHttpMessageConverter || next instanceof MappingJackson2XmlHttpMessageConverter
          || next instanceof AbstractJackson2HttpMessageConverter || next instanceof GsonHttpMessageConverter) {
        continue;
      }
      converters.add(next);
    }

    for (HttpMessageConverter<?> next : this.converters) {
      if (next instanceof StringHttpMessageConverter) {
        converters.add(1, next);
      }
    }

    converters.add(new InputStreamHttpMessageConverter());
    converters.add(new BufferedImageHttpMessageConverterCustom());

    for (HttpMessageConverter<?> next : this.converters) {
      if (next instanceof AbstractJackson2HttpMessageConverter) {
//        List<MediaType> supportedMediaTypes = new ArrayList<MediaType>(((AbstractJackson2HttpMessageConverter) next).getSupportedMediaTypes());
//        supportedMediaTypes.add(MediaType.APPLICATION_JSON_UTF8);
//        ((AbstractJackson2HttpMessageConverter) next).setSupportedMediaTypes(supportedMediaTypes);
        converters.add(next);
      }
      if (next instanceof GsonHttpMessageConverter) {
        converters.add(next);
      }
    }
    converters.add(new Jaxb2RootElementHttpMessageConverter());
    return new HttpMessageConverters(false, converters);
  }

  @Configuration
  @ConditionalOnClass({ StringHttpMessageConverter.class, Jsoup.class })
  protected static class StringHttpMessageConverterConfigurationCustom {
    private final HttpEncodingProperties encodingProperties;

    protected StringHttpMessageConverterConfigurationCustom(HttpEncodingProperties encodingProperties) {
      this.encodingProperties = encodingProperties;
    }

    @Bean
    @Primary
    public StringHttpMessageConverter stringHttpMessageConverter() {
      StringHttpMessageConverter converter = new StringHttpMessageConverterCustom(this.encodingProperties.getCharset());
      converter.setWriteAcceptCharset(false);
      return converter;
    }
  }

}
