package org.springframework.boot.autoconfigure.session;

import org.springframework.beans.factory.BeanClassLoaderAware;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.web.ServerProperties;
import org.springframework.boot.autoconfigure.web.ServerProperties.Session.Cookie;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.convert.ConversionService;
import org.springframework.core.convert.support.GenericConversionService;
import org.springframework.core.serializer.support.DeserializingConverter;
import org.springframework.core.serializer.support.SerializationFailedException;
import org.springframework.core.serializer.support.SerializingConverter;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.JdkSerializationRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.SerializationException;
import org.springframework.session.Session;
import org.springframework.session.web.http.CookieHttpSessionStrategy;
import org.springframework.session.web.http.CookieSerializer;
import org.springframework.session.web.http.DefaultCookieSerializer;
import org.springframework.session.web.http.HeaderHttpSessionStrategy;

@Configuration
@ConditionalOnClass(Session.class)
@ConditionalOnWebApplication
@AutoConfigureAfter(SessionAutoConfiguration.class)
public class SessionAutoConfigurationAfter implements BeanClassLoaderAware {
  private ClassLoader classLoader;

  /**
   * DEFAULT
   */
  @Configuration
  @ConditionalOnProperty(prefix = "session.cookie", name = "enabled", havingValue = "true", matchIfMissing = true)
  public static class CookieAutoConfiguration {
    @Bean
    @ConfigurationProperties("session.http-session-strategy")
    public CookieHttpSessionStrategy httpSessionStrategy(ObjectProvider<CookieSerializer> objectProvider) {
      CookieHttpSessionStrategy cookieHttpSessionStrategy = new CookieHttpSessionStrategy();
      CookieSerializer cookieSerializer = objectProvider.getIfAvailable();
      if (cookieSerializer != null) {
        cookieHttpSessionStrategy.setCookieSerializer(cookieSerializer);
      }
      return cookieHttpSessionStrategy;
    }

    @Bean
    @ConfigurationProperties("session.cookie-serializer")
    @ConditionalOnProperty(prefix = "session.cookie-serializer", name = "enabled", matchIfMissing = true)
    public DefaultCookieSerializer cookieSerializer(ServerProperties serverProperties) {
      DefaultCookieSerializer defaultCookieSerializer = new DefaultCookieSerializer();
      Cookie cookie = serverProperties.getSession().getCookie();
      if (cookie.getName() != null) {
        defaultCookieSerializer.setCookieName(cookie.getName());
      }
      if (cookie.getDomain() != null) {
        defaultCookieSerializer.setDomainName(cookie.getDomain());
      }
      if (cookie.getPath() != null) {
        defaultCookieSerializer.setCookiePath(cookie.getPath());
      }
      if (cookie.getHttpOnly() != null) {
        defaultCookieSerializer.setUseHttpOnlyCookie(cookie.getHttpOnly());
      }
      if (cookie.getSecure() != null) {
        defaultCookieSerializer.setUseSecureCookie(cookie.getSecure());
      }
      if (cookie.getMaxAge() != null) {
        defaultCookieSerializer.setCookieMaxAge(cookie.getMaxAge());
      }
      if (cookie.getComment() != null) {
        // defaultCookieSerializer.setComment(cookie.getComment());
      }
      return defaultCookieSerializer;
    }
  }

  @Configuration
  @ConditionalOnProperty(prefix = "session.cookie", name = "enabled", havingValue = "false", matchIfMissing = false)
  public static class HeaderAutoConfiguration {
    @Bean
    @ConfigurationProperties("session.http-session-strategy")
    public HeaderHttpSessionStrategy httpSessionStrategy() {
      HeaderHttpSessionStrategy headerHttpSessionStrategy = new HeaderHttpSessionStrategy();
      return headerHttpSessionStrategy;
    }
  }

  @Configuration
  @ConditionalOnClass(RedisTemplate.class)
  @ConditionalOnBean(RedisConnectionFactory.class)
  public static class RedisSessionConfigurationCustom implements BeanClassLoaderAware {
    private ClassLoader classLoader;

    @Bean
    public RedisSerializer<Object> springSessionDefaultRedisSerializer() {
      return new JdkSerializationRedisSerializer(classLoader != null ? classLoader : this.getClass().getClassLoader()) {
        @Override
        public Object deserialize(byte[] bytes) {
          try {
            return super.deserialize(bytes);
          }
          catch (SerializationException e) {
            return null;
          }
        }

        @Override
        public byte[] serialize(Object object) {
          try {
            return super.serialize(object);
          }
          catch (SerializationException e) {
            return new byte[] {};
          }
        }
      };
    }

    @Override
    public void setBeanClassLoader(ClassLoader classLoader) {
      this.classLoader = classLoader;
    }
  }

  /**
   * {@code byte[].class, Object.class} {@code Object.class, byte[].class}
   * @see org.springframework.core.convert.support.GenericConversionService#addConverter(Class,
   * Class, org.springframework.core.convert.converter.Converter)
   */
  @Bean
  public ConversionService springSessionConversionService() {
    GenericConversionService genericConversionService = new GenericConversionService();
    genericConversionService.addConverter(Object.class, byte[].class, new SerializingConverter() {
      @Override
      public byte[] convert(Object source) {
        try {
          return super.convert(source);
        }
        catch (SerializationFailedException e) {
          return new byte[] {};
        }
      }
    });
    genericConversionService.addConverter(byte[].class, Object.class, new DeserializingConverter(classLoader != null ? classLoader : this.getClass().getClassLoader()) {
      @Override
      public Object convert(byte[] source) {
        try {
          return super.convert(source);
        }
        catch (SerializationFailedException e) {
          return null;
        }
      }
    });
    return genericConversionService;
  }

  @Override
  public void setBeanClassLoader(ClassLoader classLoader) {
    this.classLoader = classLoader;
  }
}
