package org.springframework.boot.autoconfigure.orm.jpa;

import java.time.ZoneOffset;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TimeZone;
import java.util.TreeSet;

import javax.persistence.EntityManager;
import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionMessage;
import org.springframework.boot.autoconfigure.condition.ConditionMessage.Style;
import org.springframework.boot.autoconfigure.condition.ConditionOutcome;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.SpringBootCondition;
import org.springframework.boot.autoconfigure.jdbc.DataSourceAutoConfiguration;
import org.springframework.boot.autoconfigure.orm.jpa.HibernateJpaAutoConfiguration.HibernateEntityManagerCondition;
import org.springframework.boot.autoconfigure.orm.jpa.HibernateJpaSearchAutoConfiguration.HibernateFullTextEntityManagerCondition;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ConditionContext;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.core.type.AnnotatedTypeMetadata;
import org.springframework.jdbc.support.JdbcUtils;
import org.springframework.jdbc.support.MetaDataAccessException;
import org.springframework.orm.jpa.LocalContainerEntityManagerFactoryBean;
import org.springframework.util.ClassUtils;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StopWatch;
import org.springframework.util.SystemPropertyUtils;

/**
 * <pre>
 * entityManager.unwrap(org.hibernate.Session.class).doWork(new org.hibernate.jdbc.Work() {
 *   public void execute(Connection connection) throws SQLException {
 *     try {
 *       stopWatch.start(String.valueOf(JdbcUtils.extractDatabaseMetaData(new SingleConnectionDataSource(connection, true), "getURL")));
 *     }
 *     catch (IllegalStateException | MetaDataAccessException e) {
 *       throw new SQLException(e);
 *     }
 *   }
 * });
 * </pre>
 * 
 * @see org.springframework.boot.autoconfigure.flyway.FlywayAutoConfiguration
 */
/**
 * <pre>
 * entityManager.unwrap(org.hibernate.Session.class).doWork(new org.hibernate.jdbc.Work() {
 *   public void execute(Connection connection) throws SQLException {
 *     try {
 *       stopWatch.start(String.valueOf(JdbcUtils.extractDatabaseMetaData(new SingleConnectionDataSource(connection, true), "getURL")));
 *     }
 *     catch (IllegalStateException | MetaDataAccessException e) {
 *       throw new SQLException(e);
 *     }
 *   }
 * });
 * </pre>
 * 
 * @see org.springframework.boot.autoconfigure.flyway.FlywayAutoConfiguration
 */
@Configuration
@ConditionalOnClass({ LocalContainerEntityManagerFactoryBean.class, EntityManager.class })
// @ConditionalOnBean({ EntityManagerFactory.class })
@AutoConfigureAfter({ DataSourceAutoConfiguration.class, HibernateJpaAutoConfiguration.class })
@Conditional({ HibernateFullTextEntityManagerCondition.class, HibernateEntityManagerCondition.class })
public class HibernateJpaSearchAutoConfiguration {
  @Bean
  @ConfigurationProperties("spring.jpa.hibernate.search")
  @ConditionalOnProperty(prefix = "spring.jpa.hibernate.search", name = "enabled", matchIfMissing = true)
  public HibernateJpaSearchListener hibernateJpaSearchListener() {
    return new HibernateJpaSearchListener();
  }

  @Order(Ordered.HIGHEST_PRECEDENCE + 20)
  static class HibernateFullTextEntityManagerCondition extends SpringBootCondition {
    private static String[] CLASS_NAMES = { "org.hibernate.search.jpa.FullTextEntityManager" };

    @Override
    public ConditionOutcome getMatchOutcome(ConditionContext context, AnnotatedTypeMetadata metadata) {
      ConditionMessage.Builder message = ConditionMessage.forCondition("FullTextEntityManager");
      for (String className : CLASS_NAMES) {
        if (ClassUtils.isPresent(className, context.getClassLoader())) {
          return ConditionOutcome.match(message.found("class").items(Style.QUOTE, className));
        }
      }
      return ConditionOutcome.noMatch(message.didNotFind("class", "classes").items(Style.QUOTE, Arrays.asList(CLASS_NAMES)));
    }
  }
}

class HibernateJpaSearchListener implements ApplicationListener<ApplicationReadyEvent> {
  protected final Log logger = LogFactory.getLog(getClass());
  private String lineSeparator = SystemPropertyUtils.resolvePlaceholders("${line.separator:\r\n}");
  private Set<String> excludes = new TreeSet<String>(String.CASE_INSENSITIVE_ORDER);

  /**
   * @see java.time.ZoneOffset#UTC
   * @see org.hibernate.cfg.Environment
   */
  @Override
  public void onApplicationEvent(ApplicationReadyEvent event) {
    Map<String, EntityManager> map = event.getApplicationContext().getBeansOfType(EntityManager.class);
    if (CollectionUtils.isEmpty(map)) {
      return;
    }
    Calendar calendar = Calendar.getInstance(TimeZone.getTimeZone(ZoneOffset.UTC));
    calendar.setTimeInMillis(event.getTimestamp());
    StopWatch stopWatch = new StopWatch(javax.xml.bind.DatatypeConverter.printDateTime(calendar));
    for (Entry<String, EntityManager> entry : map.entrySet()) {
      EntityManager entityManager = entry.getValue();
      Object persistenceUnitName = entityManager.getEntityManagerFactory().getProperties().get(org.hibernate.jpa.AvailableSettings.PERSISTENCE_UNIT_NAME);
      if (excludes.contains(persistenceUnitName)) {
        continue;
      }
      Object dataSource = entityManager.getEntityManagerFactory().getProperties().get(org.hibernate.cfg.AvailableSettings.DATASOURCE);
      if (dataSource instanceof DataSource) {
        try {
          dataSource = JdbcUtils.extractDatabaseMetaData((DataSource) dataSource, "getURL");
        }
        catch (MetaDataAccessException e) {
          // ignore
        }
      }
      StringBuilder stringBuilder = new StringBuilder(entry.getKey());
      if (persistenceUnitName instanceof String) {
        stringBuilder.append(lineSeparator);
        stringBuilder.append("persistenceUnitName: ");
        stringBuilder.append(persistenceUnitName);
      }
      if (dataSource instanceof String) {
        stringBuilder.append(lineSeparator);
        stringBuilder.append("dataSource: ");
        stringBuilder.append(dataSource);
      }
      stopWatch.start(new String(stringBuilder));
      try {
        org.hibernate.search.jpa.Search.getFullTextEntityManager(entityManager).createIndexer().startAndWait();
      }
      catch (NoClassDefFoundError | IllegalArgumentException | InterruptedException e) {
        if (logger.isTraceEnabled()) {
          logger.trace("An error occurred trying to build the search index: " + e, e);
        }
        else if (logger.isWarnEnabled()) {
          logger.warn("An error occurred trying to build the search index: " + e);
        }
      }
      finally {
        stopWatch.stop();
      }
    }
    if (logger.isInfoEnabled()) {
      logger.info(stopWatch.prettyPrint());
    }
    return;
  }

  public Set<String> getExcludes() {
    return excludes;
  }

  public void setExcludes(Set<String> excludes) {
    this.excludes = excludes;
  }
}