package org.springframework.boot.autoconfigure.orm.jpa;

import java.lang.reflect.Method;
import java.util.LinkedHashMap;
import java.util.Map;

import javax.persistence.EntityManager;
import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.jdbc.DataSourceAutoConfiguration;
import org.springframework.boot.autoconfigure.orm.jpa.HibernateJpaAutoConfiguration.HibernateEntityManagerCondition;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.orm.jpa.JpaVendorAdapter;
import org.springframework.orm.jpa.LocalContainerEntityManagerFactoryBean;
import org.springframework.orm.jpa.vendor.AbstractJpaVendorAdapter;
import org.springframework.orm.jpa.vendor.Database;
import org.springframework.orm.jpa.vendor.HibernateJpaVendorAdapter;
import org.springframework.util.ReflectionUtils;

@Configuration
@ConditionalOnClass({ LocalContainerEntityManagerFactoryBean.class, EntityManager.class })
@Conditional(HibernateEntityManagerCondition.class)
@AutoConfigureAfter({ DataSourceAutoConfiguration.class, HibernateJpaAutoConfiguration.class })
@ConfigurationProperties(prefix = "spring.jpa.hibernate")
public class HibernateJpaAutoConfigurationAfter {
  protected final Log logger = LogFactory.getLog(getClass());

  private Map<Database, Class<? extends org.hibernate.dialect.Dialect>> dialect = new LinkedHashMap<Database, Class<? extends org.hibernate.dialect.Dialect>>();

  /**
   * <code>
   * throw new IllegalStateException("Not supported database: " + database);
   * </code>
   * @see java.lang.UnsupportedOperationException#UnsupportedOperationException()
   */
  @Bean
  public JpaVendorAdapter jpaVendorAdapter(JpaProperties jpaProperties, DataSource dataSource) {
    AbstractJpaVendorAdapter adapter = new HibernateJpaVendorAdapter() {
      @Override
      protected Class<?> determineDatabaseDialectClass(Database database) {
        Class<?> dialectClass = dialect.get(database);
        if (dialectClass == null) {
          dialectClass = super.determineDatabaseDialectClass(database);
        }
        return dialectClass;
        // switch (database) {
        // case DB2:
        // return org.hibernate.dialect.DB2Dialect.class;
        // case DERBY:
        // return org.hibernate.dialect.DerbyTenSevenDialect.class;
        // case H2:
        // return org.hibernate.dialect.H2Dialect.class;
        // case HSQL:
        // return org.hibernate.dialect.HSQLDialect.class;
        // case INFORMIX:
        // return org.hibernate.dialect.InformixDialect.class;
        // case MYSQL:
        // return org.hibernate.dialect.MySQL5Dialect.class;
        // case ORACLE:
        // return org.hibernate.dialect.Oracle9iDialect.class;
        // case POSTGRESQL:
        // return org.hibernate.dialect.PostgreSQL9Dialect.class;
        // case SQL_SERVER:
        // return org.hibernate.dialect.SQLServer2008Dialect.class;
        // case SYBASE:
        // return org.hibernate.dialect.SybaseDialect.class;
        // default:
        // throw new IllegalArgumentException("Not supported database: " + database);
        // }
      }
    };
    adapter.setDatabase(jpaProperties.determineDatabase(dataSource));
    adapter.setShowSql(jpaProperties.isShowSql());
    adapter.setGenerateDdl(jpaProperties.isGenerateDdl());
    adapter.setDatabasePlatform(jpaProperties.getDatabasePlatform());
    return adapter;
  }

  @Bean
  public Database database(JpaVendorAdapter jpaVendorAdapter) {
    if (jpaVendorAdapter instanceof AbstractJpaVendorAdapter) {
      Method method = ReflectionUtils.findMethod(jpaVendorAdapter.getClass(), "getDatabase");
      if (method != null) {
        ReflectionUtils.makeAccessible(method);
        Object object = ReflectionUtils.invokeMethod(method, jpaVendorAdapter);
        if (object instanceof Database) {
          if (logger.isInfoEnabled()) {
            logger.info("Database: " + object);
          }
          return (Database) object;
        }
      }
    }
    return Database.DEFAULT;
  }

  public Map<Database, Class<? extends org.hibernate.dialect.Dialect>> getDialect() {
    return dialect;
  }

  public void setDialect(Map<Database, Class<? extends org.hibernate.dialect.Dialect>> dialect) {
    this.dialect = dialect;
  }
}
