package org.springframework.boot.autoconfigure.network;

import java.util.List;

import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.web.WebMvcAutoConfiguration;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.network.LiteNetworkResolver;
import org.springframework.network.NetworkHandlerMethodArgumentResolver;
import org.springframework.network.NetworkResolver;
import org.springframework.network.NetworkResolverHandlerInterceptor;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;

@Configuration
@ConditionalOnClass({ NetworkResolverHandlerInterceptor.class, NetworkHandlerMethodArgumentResolver.class })
@AutoConfigureAfter(WebMvcAutoConfiguration.class)
@ConditionalOnWebApplication
public class NetworkResolverAutoConfiguration {

  @Bean
  @ConfigurationProperties(prefix = "spring.network.networkresolver")
  @ConditionalOnMissingBean(NetworkResolver.class)
  public NetworkResolver networkResolver() {
    return new LiteNetworkResolver();
  }

  @Bean
  @ConditionalOnBean(NetworkResolver.class)
  @ConditionalOnMissingBean(NetworkResolverHandlerInterceptor.class)
  public NetworkResolverHandlerInterceptor networkResolverHandlerInterceptor(NetworkResolver networkResolver) {
    return new NetworkResolverHandlerInterceptor(networkResolver);
  }

  @Bean
  public NetworkHandlerMethodArgumentResolver networkHandlerMethodArgumentResolver() {
    return new NetworkHandlerMethodArgumentResolver();
  }

  @Configuration
  @Order(0)
  protected static class NetworkResolverMvcConfiguration extends WebMvcConfigurerAdapter {
    private NetworkResolverHandlerInterceptor networkResolverHandlerInterceptor;
    private NetworkHandlerMethodArgumentResolver networkHandlerMethodArgumentResolver;

    protected NetworkResolverMvcConfiguration(NetworkResolverHandlerInterceptor deviceResolverHandlerInterceptor, NetworkHandlerMethodArgumentResolver deviceHandlerMethodArgumentResolver) {
      this.networkResolverHandlerInterceptor = deviceResolverHandlerInterceptor;
      this.networkHandlerMethodArgumentResolver = deviceHandlerMethodArgumentResolver;
    }

    @Override
    public void addInterceptors(InterceptorRegistry registry) {
      registry.addInterceptor(this.networkResolverHandlerInterceptor);
    }

    @Override
    public void addArgumentResolvers(List<HandlerMethodArgumentResolver> argumentResolvers) {
      argumentResolvers.add(this.networkHandlerMethodArgumentResolver);
    }
  }

}
