package org.springframework.boot.autoconfigure.jdbc;

import javax.sql.DataSource;

import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.jdbc.metadata.DataSourcePoolMetadataProvider;
import org.springframework.boot.autoconfigure.jdbc.metadata.DecoratorDataSourcePoolMetadataProvider;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.jdbc.datasource.LazyConnectionDataSourceProxy;

import com.p6spy.engine.spy.P6DataSource;

import net.ttddyy.dsproxy.support.ProxyDataSource;

/**
 * @see org.springframework.boot.autoconfigure.jdbc.DataSourceAutoConfiguration
 * @see org.springframework.boot.autoconfigure.flyway.FlywayAutoConfiguration#FlywayInitializerJpaDependencyConfiguration
 */
@Configuration
@EnableConfigurationProperties(DataSourceDecoratorProperties.class)
@ConditionalOnProperty(prefix = "spring.datasource.decorator", name = "enabled", matchIfMissing = true)
@ConditionalOnBean(DataSource.class)
@AutoConfigureAfter(DataSourceAutoConfiguration.class)
public class DataSourceDecoratorAutoConfiguration {
  @Bean
  public DataSourceDecoratorPostProcessor dataSourceDecoratorBeanPostProcessor(DataSourceDecoratorProperties properties, ApplicationContext applicationContext) {
    return new DataSourceDecoratorPostProcessor(properties, applicationContext);
  }

  @Bean
  public DataSourcePoolMetadataProvider decoratorDataSourcePoolMetadataProvider() {
    return new DecoratorDataSourcePoolMetadataProvider();
  }

  @Bean
  public DataSourceDecorator lazyConnectionDataSourceProxyDecorator() {
    return new DataSourceDecorator() {
      @Override
      public DataSource decorate(DataSource dataSource) {
        return new LazyConnectionDataSourceProxy(dataSource);
      }

      @Override
      public int getOrder() {
        return Ordered.LOWEST_PRECEDENCE;
      }
    };
  }

  @Configuration
  @ConditionalOnClass(P6DataSource.class)
  protected static class P6SpyDataSourceDecoratorConfiguration {
    @Bean
    public DataSourceDecorator p6SpyDataSourceDecorator() {
      return new DataSourceDecorator() {
        @Override
        public DataSource decorate(DataSource dataSource) {
          return new P6DataSource(dataSource);
        }

        @Override
        public int getOrder() {
          return Ordered.LOWEST_PRECEDENCE - 10;
        }
      };
    }
  }

  @Configuration
  @ConditionalOnClass(ProxyDataSource.class)
  protected static class ProxyDataSourceDecoratorConfiguration {
    @Bean
    public DataSourceDecorator proxyDataSourceDecorator() {
      return new DataSourceDecorator() {
        @Override
        public DataSource decorate(DataSource dataSource) {
          return new ProxyDataSource(dataSource);
        }

        @Override
        public int getOrder() {
          return Ordered.LOWEST_PRECEDENCE - 20;
        }
      };
    }
  }

}
