package org.springframework.boot.autoconfigure.flyway;

import java.util.ArrayList;
import java.util.List;

import javax.sql.DataSource;

import org.flywaydb.core.Flyway;
import org.flywaydb.core.api.resolver.MigrationResolver;
import org.flywaydb.core.internal.dbsupport.DbSupport;
import org.flywaydb.core.internal.dbsupport.DbSupportFactory;
import org.flywaydb.core.internal.resolver.jdbc.JdbcMigrationResolver;
import org.flywaydb.core.internal.resolver.spring.SpringJdbcMigrationResolver;
import org.flywaydb.core.internal.resolver.spring.SpringJdbcMigrationResolverCustom;
import org.flywaydb.core.internal.resolver.sql.SqlMigrationResolver;
import org.flywaydb.core.internal.util.FeatureDetector;
import org.flywaydb.core.internal.util.Locations;
import org.flywaydb.core.internal.util.PlaceholderReplacer;
import org.flywaydb.core.internal.util.jdbc.JdbcUtils;
import org.flywaydb.core.internal.util.scanner.Scanner;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * <pre>
 * &#64;Bean
 * &#64;ConditionalOnClass(Flyway.class)
 * &#64;ConditionalOnBean(DataSource.class)
 * public FlywayMigrationStrategy migrationStrategy(ApplicationContext applicationContext) {
 *   return new FlywayMigrationStrategy() {
 *     &#64;Override
 *     public void migrate(Flyway flyway) {
 *       List<MigrationResolver> migrationResolver = new ArrayList<MigrationResolver>();
 *       for (Location location : new Locations(flyway.getLocations()).getLocations()) {
 *         migrationResolver.add(new SpringJdbcMigrationResolverCustom(flyway.getClassLoader(), location, applicationContext));
 *       }
 * 
 *       flyway.setResolvers(migrationResolver.toArray(new MigrationResolver[migrationResolver.size()]));
 *       flyway.setLocations(ClassUtils.classPackageAsResourcePath(getClass()));
 *       flyway.migrate();
 *     }
 *   };
 * }
 * </pre>
 */
@Configuration
@ConditionalOnClass(Flyway.class)
@ConditionalOnBean(DataSource.class)
@ConditionalOnProperty(prefix = "flyway", name = "enabled", matchIfMissing = true)
@AutoConfigureAfter(FlywayAutoConfiguration.class)
public class FlywayAutoConfigurationAfter {

  @Bean
  @ConditionalOnMissingBean(FlywayMigrationStrategy.class)
  public FlywayMigrationStrategy migrationStrategy(ApplicationContext applicationContext) {
    return new FlywayMigrationStrategy() {
      /**
       * <pre>
       * migrationResolvers.add(new JdbcMigrationResolverCustom(flyway.getClassLoader(), location, this.applicationContext));
       * migrationResolvers.add(new SpringJdbcMigrationResolverCustom(flyway.getClassLoader(), location, this.applicationContext));
       * flyway.setLocations(ClassUtils.addResourcePathToPackagePath(getClass(), "migration"))
       * </pre>
       */
      @Override
      public void migrate(Flyway flyway) {
        if (flyway.isSkipDefaultResolvers()) {
          List<MigrationResolver> migrationResolvers = new ArrayList<MigrationResolver>();
          DbSupport dbSupport = DbSupportFactory.createDbSupport(JdbcUtils.openConnection(flyway.getDataSource()), true);
          Scanner scanner = new Scanner(flyway.getClassLoader());
          Locations locations = new Locations(flyway.getLocations());
          PlaceholderReplacer placeholderReplacer = flyway.isPlaceholderReplacement() ? new PlaceholderReplacer(flyway.getPlaceholders(), flyway.getPlaceholderPrefix(), flyway.getPlaceholderSuffix())
              : PlaceholderReplacer.NO_PLACEHOLDERS;
          migrationResolvers.add(new SqlMigrationResolver(dbSupport, scanner, locations, placeholderReplacer, flyway));
          migrationResolvers.add(new JdbcMigrationResolver(scanner, locations, flyway));
          if (new FeatureDetector(scanner.getClassLoader()).isSpringJdbcAvailable()) {
            migrationResolvers.add(new SpringJdbcMigrationResolverCustom(scanner, locations, flyway, applicationContext));
            migrationResolvers.add(new SpringJdbcMigrationResolver(scanner, locations, flyway));
          }
          flyway.setResolvers(migrationResolvers.toArray(new MigrationResolver[migrationResolvers.size()]));
        }
        flyway.migrate();
      }
    };
  }
}
