package io.kotlintest

import io.kotlintest.matchers.ToleranceMatcher
import io.kotlintest.matchers.shouldBe
import org.junit.ComparisonCompactor

fun <T> be(expected: T) = equalityMatcher(expected)
fun <T> equalityMatcher(expected: T) = object : Matcher<T> {
  override fun test(value: T): Result = Result(expected == value, equalsErrorMessage(expected, value), "$value should not equal $expected")
}

fun fail(msg: String): Nothing = throw AssertionError(msg)

// -- equality functions

infix fun <T, U : T> T.shouldBe(any: U?) {
  when (any) {
    is Matcher<*> -> should(any as Matcher<T>)
    else -> {
      if (this == null && any != null)
        throw equalsError(any, this)
      if (this != any)
        throw equalsError(any, this)
    }
  }
}

infix fun <T> T.shouldNotBe(any: Any?) {
  when (any) {
    is Matcher<*> -> shouldNot(any as Matcher<T>)
    else -> shouldNot(equalityMatcher(any))
  }
}

// -- matcher functions

infix fun <T> T.shouldHave(matcher: Matcher<T>) = should(matcher)
infix fun <T> T.should(matcher: Matcher<T>) {
  val result = matcher.test(this)
  if (!result.passed)
    throw AssertionError(result.failureMessage)
}

infix fun <T> T.shouldNotHave(matcher: Matcher<T>) = shouldNot(matcher)
infix fun <T> T.shouldNot(matcher: Matcher<T>) {
  val result = matcher.test(this)
  if (result.passed)
    throw AssertionError(result.negatedFailureMessage)
}

infix fun <T> T.should(matcher: (T) -> Unit) = matcher(this)


// -- specialized overrides of shouldBe --

infix fun Double.shouldBe(other: Double) = should(ToleranceMatcher(other, 0.0))

// https://stackoverflow.com/questions/10934743/formatting-output-so-that-intellij-idea-shows-diffs-for-two-texts
// https://github.com/JetBrains/intellij-community/blob/3f7e93e20b7e79ba389adf593b3b59e46a3e01d1/plugins/testng/src/com/theoryinpractice/testng/model/TestProxy.java#L50
infix fun String.shouldBe(other: String) {
  if (this != other) {
    throw AssertionError(ComparisonCompactor.getMessage(other, this))
  }
}

infix fun BooleanArray.shouldBe(other: BooleanArray) {
  val expected = other.toList()
  val actual = this.toList()
  if (actual != expected)
    throw equalsError(expected, actual)
}

infix fun IntArray.shouldBe(other: IntArray) {
  val expected = other.toList()
  val actual = this.toList()
  if (actual != expected)
    throw equalsError(expected, actual)
}

infix fun DoubleArray.shouldBe(other: DoubleArray) {
  val expected = other.toList()
  val actual = this.toList()
  if (actual != expected)
    throw equalsError(expected, actual)
}

infix fun LongArray.shouldBe(other: LongArray) {
  val expected = other.toList()
  val actual = this.toList()
  if (actual != expected)
    throw equalsError(expected, actual)
}

infix fun <T> Array<T>.shouldBe(other: Array<T>) {
  val expected = other.toList()
  val actual = this.toList()
  if (actual != expected)
    throw equalsError(expected, actual)
}

private fun equalsError(expected: Any?, actual: Any?) = AssertionError(equalsErrorMessage(expected, actual))
private fun equalsErrorMessage(expected: Any?, actual: Any?) = "expected: $expected but was: $actual"

// -- deprecated dsl

@Deprecated("shouldEqual is deprecated in favour of shouldBe", ReplaceWith("shouldBe(any)"))
infix fun <T> T.shouldEqual(any: Any?) = shouldBe(any)