package io.julian.common.widget;

import android.content.Context;
import android.support.annotation.StringRes;
import android.util.AttributeSet;
import android.view.Gravity;
import android.view.View;
import android.widget.Button;
import android.widget.FrameLayout;
import android.widget.LinearLayout;
import android.widget.ProgressBar;
import android.widget.TextView;

import io.julian.common.R;

/**
 * @author Zhu Liang
 */

public class LoadingLayout extends FrameLayout {

    public final static int SUCCESS = 0;//加载成功
    public final static int EMPTY = 1;//无数据
    public final static int ERROR = 2;//错误
    public final static int NO_NETWORK = 3;//无网络
    public final static int LOADING = 4;//加载中

    private OnReloadListener mOnReloadListener;
    private static Config sDefaultConfig = new Config();
    private Config mConfig;

    private int mStatus = LOADING;

    private Context mContext;

    private View mContentView;
    private ProgressBar mProgressBar;
    private TextView mTextView;
    private Button mButton;

    public LoadingLayout(Context context) {
        this(context, null);
    }

    public LoadingLayout(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public LoadingLayout(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        mContext = context;
    }

    @Override
    protected void onFinishInflate() {
        super.onFinishInflate();
        if (getChildCount() > 1) {
            throw new IllegalStateException("loadinglayout_ can host only one direct child");
        }

        mContentView = getChildAt(0);
        mContentView.setVisibility(GONE);

        build();
    }

    public void setOnReloadListener(OnReloadListener onReloadListener) {
        mOnReloadListener = onReloadListener;
    }

    public void setStatus(int status) {
        mStatus = status;

        Config config = mConfig != null ? mConfig : sDefaultConfig;

        switch (mStatus) {
            case SUCCESS:
                mContentView.setVisibility(VISIBLE);
                mProgressBar.setVisibility(GONE);
                mTextView.setVisibility(GONE);
                mButton.setVisibility(GONE);
                break;
            case EMPTY:
                mContentView.setVisibility(GONE);
                mProgressBar.setVisibility(GONE);
                mTextView.setVisibility(VISIBLE);
                mButton.setVisibility(GONE);

                mTextView.setText(config.getEmptyTextId());
                break;
            case ERROR:
                mContentView.setVisibility(GONE);
                mProgressBar.setVisibility(GONE);
                mTextView.setVisibility(VISIBLE);
                mButton.setVisibility(VISIBLE);

                mTextView.setText(config.getErrorTextId());
                mButton.setText(config.getErrorButtonId());
                break;
            case NO_NETWORK:
                mContentView.setVisibility(GONE);
                mProgressBar.setVisibility(GONE);
                mTextView.setVisibility(VISIBLE);
                mButton.setVisibility(VISIBLE);

                mTextView.setText(config.getNoNetworkTextId());
                mButton.setText(config.getNoNetworkButtonId());
                break;
            case LOADING:
                mContentView.setVisibility(GONE);
                mProgressBar.setVisibility(VISIBLE);
                mTextView.setVisibility(VISIBLE);
                mButton.setVisibility(GONE);

                mTextView.setText(config.getLoadingTextId());
                break;
        }
    }

    public LoadingLayout setLoadingText(@StringRes int id) {
        initConfig();
        mConfig.setLoadingText(id);
        return this;
    }

    public LoadingLayout setErrorText(@StringRes int id) {
        initConfig();
        mConfig.setErrorText(id);
        return this;
    }

    public LoadingLayout setErrorButton(@StringRes int id) {
        initConfig();
        mConfig.setErrorButton(id);
        return this;
    }

    public LoadingLayout setEmptyText(@StringRes int id) {
        initConfig();
        mConfig.setEmptyText(id);
        return this;
    }

    public LoadingLayout setNoNetworkText(@StringRes int id) {
        initConfig();
        mConfig.setNoNetworkText(id);
        return this;
    }

    public LoadingLayout setNoNetworkButton(@StringRes int id) {
        initConfig();
        mConfig.setNoNetworkButton(id);
        return this;
    }

    private void initConfig() {
        if (mConfig == null) {
            mConfig = new Config();
        }
    }

    private void build() {

        LayoutParams lp = new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT);
        lp.gravity = Gravity.CENTER;
        LinearLayout linearLayout = new LinearLayout(mContext);
        linearLayout.setLayoutParams(lp);
        linearLayout.setGravity(Gravity.CENTER);
        linearLayout.setOrientation(LinearLayout.VERTICAL);

        LinearLayout.LayoutParams clp = new LinearLayout.LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
        mProgressBar = new ProgressBar(mContext);
        mProgressBar.setLayoutParams(clp);
        linearLayout.addView(mProgressBar);

        clp = new LinearLayout.LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
        mTextView = new TextView(mContext);
        mTextView.setLayoutParams(clp);
        linearLayout.addView(mTextView);

        clp = new LinearLayout.LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
        mButton = new Button(mContext);
        mButton.setLayoutParams(clp);
        linearLayout.addView(mButton);

        mButton.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mOnReloadListener != null) {
                    mOnReloadListener.onReload(mStatus, v);
                }
            }
        });

        addView(linearLayout);

        setStatus(mStatus);
    }

    public interface OnReloadListener {
        void onReload(int status, View view);
    }

    public static class Config {

        private int mLoadingTextId = R.string.loadinglayout_loading_text;
        private int mErrorTextId = R.string.loadinglayout_error_text;
        private int mErrorButtonId = R.string.loadinglayout_error_button;
        private int mEmptyTextId = R.string.loadinglayout_empty_text;
        private int mNoNetworkTextId = R.string.loadinglayout_no_network_text;
        private int mNoNetworkButtonId = R.string.loadinglayout_no_network_button;

        public Config setLoadingText(@StringRes int id) {
            mLoadingTextId = id;
            return this;
        }

        public Config setErrorText(@StringRes int id) {
            mErrorTextId = id;
            return this;
        }

        public Config setErrorButton(@StringRes int id) {
            mErrorButtonId = id;
            return this;
        }

        public Config setEmptyText(@StringRes int id) {
            mEmptyTextId = id;
            return this;
        }

        public Config setNoNetworkText(@StringRes int id) {
            mNoNetworkTextId = id;
            return this;
        }

        public Config setNoNetworkButton(@StringRes int id) {
            mNoNetworkButtonId = id;
            return this;
        }

        public int getLoadingTextId() {
            return mLoadingTextId;
        }

        public void setLoadingTextId(int loadingTextId) {
            mLoadingTextId = loadingTextId;
        }

        public int getErrorTextId() {
            return mErrorTextId;
        }

        public void setErrorTextId(int errorTextId) {
            mErrorTextId = errorTextId;
        }

        public int getErrorButtonId() {
            return mErrorButtonId;
        }

        public void setErrorButtonId(int errorButtonId) {
            mErrorButtonId = errorButtonId;
        }

        public int getEmptyTextId() {
            return mEmptyTextId;
        }

        public void setEmptyTextId(int emptyTextId) {
            mEmptyTextId = emptyTextId;
        }

        public int getNoNetworkTextId() {
            return mNoNetworkTextId;
        }

        public void setNoNetworkTextId(int noNetworkTextId) {
            mNoNetworkTextId = noNetworkTextId;
        }

        public int getNoNetworkButtonId() {
            return mNoNetworkButtonId;
        }

        public void setNoNetworkButtonId(int noNetworkButtonId) {
            mNoNetworkButtonId = noNetworkButtonId;
        }
    }
}
