//
// (C) ITculate, Inc. 2015-2017
// All rights reserved
// Licensed under MIT License (see LICENSE)
//

package com.itculate.sdk.types;

import org.json.JSONObject;

public abstract class DataType {
    static int DEFAULT_IMPORTANCE = 100;
    static ResamplingMethod DEFAULT_RESAMPLING_METHOD = ResamplingMethod.MEAN;

    private String label;
    private int importance;
    private String description;
    private boolean visible;
    private ResamplingMethod resamplingMethod;
    private String type;

    protected DataType(Builder builder) {
        this.label = builder.label;
        this.importance = builder.importance;
        this.description = builder.description;
        this.visible = builder.visible;
        this.resamplingMethod = builder.resamplingMethod;

        String simpleName = this.getClass().getSimpleName();
        if (!simpleName.endsWith("DataType"))
            throw new IllegalStateException("Bad DataType class name, " + simpleName + " must end with DataType");

        this.type = simpleName.substring(0, simpleName.length() - 8);
    }

    abstract public static class Builder<T extends DataType> {
        private String label = null;
        private int importance = DEFAULT_IMPORTANCE;
        private String description = null;
        private boolean visible = true;
        private ResamplingMethod resamplingMethod = DEFAULT_RESAMPLING_METHOD;

        public Builder<T> label(String label) {
            this.label = label;
            return this;
        }

        public Builder<T> importance(int importance) {
            this.importance = importance;
            return this;
        }

        public Builder<T> resamplingMethod(ResamplingMethod resamplingMethod) {
            this.resamplingMethod = resamplingMethod;
            return this;
        }

        public Builder<T> description(String description) {
            this.description = description;
            return this;
        }

        public Builder<T> visible(boolean visible) {
            this.visible = visible;
            return this;
        }

        abstract public T build();
    }

    public JSONObject toJson() {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("topology", this.type);
        jsonObject.put("importance", this.importance);
        jsonObject.put("visible", this.visible);
        if (this.description != null)
            jsonObject.put("description", this.description);
        if (this.label != null)
            jsonObject.put("label", this.label);

        return jsonObject;
    }


}
