//
// (C) ITculate, Inc. 2015-2017
// All rights reserved
// Licensed under MIT License (see LICENSE)
//

package com.itculate.sdk.provider;

import com.itculate.sdk.ITculateApiException;
import org.apache.http.HttpHost;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.AuthCache;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.entity.AbstractHttpEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.impl.client.BasicAuthCache;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.json.JSONObject;
import sun.font.TrueTypeFont;

import java.io.*;
import java.net.InetAddress;
import java.net.URI;
import java.nio.file.Paths;
import java.util.Properties;

/**
 * A simple client for submitting topologies via the ITculate API.
 */
public class SynchronousApiProvider implements Provider {

    private String apiKey = null;
    private String apiSecret = null;
    private String serverUrl = DEFAULT_SYNCHRONOUS_API_URL;
    protected String credentialsRole = DEFAULT_CREDENTIALS_ROLE;

    public SynchronousApiProvider(String serverUrl, String apiKey, String apiSecret) {

        if (serverUrl != null)
            this.serverUrl = serverUrl;

        this.apiKey = apiKey;
        this.apiSecret = apiSecret;

        if (this.apiKey == null && this.apiSecret == null)
            loadCredentialsFromFile(this.credentialsRole);
    }

    public SynchronousApiProvider(String credentialsRole, String serverUrl) {
        if (credentialsRole != null)
            this.credentialsRole = credentialsRole;

        if (serverUrl != null)
            this.serverUrl = serverUrl;


        loadCredentialsFromFile(this.credentialsRole);
    }

    private void loadCredentialsFromFile(String credentialsRole) {

        String path = Paths.get(System.getProperty("user.home"), ".itculate", "credentials").toString();

        String role = "[" + credentialsRole + "]";

        try {
            if (new File(path).exists()) {
                BufferedReader fr = new BufferedReader(new FileReader(path));

                boolean roleFound = false;
                String input;
                while ((input = fr.readLine()) != null) {
                    if (input.isEmpty()) {
                        continue;
                    }

                    if (input.equals(role)) {
                        roleFound = true;
                        continue;
                    }

                    if (roleFound) {
                        if (input.startsWith("api_key="))
                            this.apiKey = input.substring(8);
                        else if (input.startsWith("api_secret="))
                            this.apiSecret = input.substring(11);
                    }

                    if ((this.apiKey != null) && (this.apiSecret != null))
                        break;
                }
                fr.close();
            }
        } catch (IOException e) {
            throw new IllegalStateException("Failed to load credentials [" + credentialsRole + "] from " + path);
        }

        if ((this.apiKey == null) || (this.apiSecret == null)) {
            throw new IllegalStateException("Failed to load credentials [" + credentialsRole + "] from " + path);
        }
    }

    /**
     * Submits a custom ITculate event via the ITculate API.
     *
     * @throws ITculateApiException if failed.
     */
    public void flush(String collectorId, JSONObject jsonObject) throws ITculateApiException {

        if (this.serverUrl == null || this.apiKey == null || this.apiSecret == null) {
            throw new IllegalStateException("serverUrl, apiKey and apiSecret cannot be null");
        }

        if (jsonObject == null) {
            throw new IllegalStateException("jsonObject must not be null");
        }

        if (collectorId == null) {
            throw new IllegalStateException("collectorId must not be null");
        }

        try {
            JSONObject jsonEntity = new JSONObject();
            jsonEntity.put("compressed_payload", Util.compressForJson(jsonObject));
            jsonEntity.put("collector_id", collectorId);
            jsonEntity.put("collector_version", "0.1.1");
            jsonEntity.put("host", InetAddress.getLocalHost().getHostAddress());
            String payload = jsonEntity.toString();

            URI uri = URI.create(this.serverUrl);
            doPost(uri, payload);
        } catch (Exception e) {
            throw new ITculateApiException("Failed to flush payload collectorId " + collectorId, e);
        }
    }

    protected void doPost(URI apiServerUri, String payload) throws Exception {
        String host = apiServerUri.getHost();
        int port = apiServerUri.getPort();
        String scheme = apiServerUri.getScheme();


        HttpHost targetHost = new HttpHost(host, port, scheme);
        CredentialsProvider credsProvider = new BasicCredentialsProvider();
        UsernamePasswordCredentials credentials = new UsernamePasswordCredentials(apiKey, apiSecret);
        AuthScope authScope = AuthScope.ANY;
        credsProvider.setCredentials(authScope, credentials);

        // Create AuthCache instance
        AuthCache authCache = new BasicAuthCache();
        // Generate BASIC scheme object and add it to the local auth cache
        BasicScheme basicAuth = new BasicScheme();
        authCache.put(targetHost, basicAuth);


        // Add AuthCache to the execution context
        HttpClientContext context = HttpClientContext.create();
        context.setCredentialsProvider(credsProvider);
        context.setAuthCache(authCache);

        RequestConfig globalConfig = RequestConfig.custom().setCookieSpec(CookieSpecs.DEFAULT).build();
        CloseableHttpClient httpclient = HttpClients.custom().setDefaultRequestConfig(globalConfig).build();

        AbstractHttpEntity entity = new StringEntity(payload);

        entity.setContentType("application/json");

        HttpPost httpPost = new HttpPost(API_END_POINT);
        httpPost.setEntity(entity);

        CloseableHttpResponse response = httpclient.execute(targetHost, httpPost, context);

        Util.checkResponse(response);
    }
}
