//
// (C) ITculate, Inc. 2015-2017
// All rights reserved
// Licensed under MIT License (see LICENSE)
//

package com.itculate.sdk;

import com.itculate.sdk.types.DataType;
import com.itculate.sdk.types.NumberTypedValue;
import org.json.JSONArray;
import org.json.JSONObject;

import java.util.*;

public class Payload {

    private TopologyPayload topologyPayload = new TopologyPayload();
    private EventsPayload eventsPayload = new EventsPayload();
    private TimeseriesPayload timeseriesPayload = new TimeseriesPayload();
    private DictionaryPayload dictionaryPayload = new DictionaryPayload();

    private String collectorId;

    public Payload(String collectorId) {
        this.collectorId = collectorId;
    }

    public String getCollectorId() {
        return collectorId;
    }


    final public JSONObject toJson() {
        TopologyPayload localTopologyPayload = topologyPayload;
        EventsPayload localEventsPayload = eventsPayload;
        TimeseriesPayload localTimeseriesPayload = timeseriesPayload;
        DictionaryPayload localDictionaryPayload = dictionaryPayload;

        synchronized (this) {
            topologyPayload = new TopologyPayload();
            eventsPayload = new EventsPayload();
            timeseriesPayload = new TimeseriesPayload();
            dictionaryPayload = new DictionaryPayload();
        }

        JSONObject jsonObject = new JSONObject();
        localTopologyPayload.doFillJson(jsonObject);
        localEventsPayload.doFillJson(jsonObject, collectorId);
        localTimeseriesPayload.doFillJson(jsonObject);
        localDictionaryPayload.doFillJson(jsonObject);
        return jsonObject;
    }

    @Override
    public String toString() {
        return this.toJson().toString();
    }

    public void add(Vertex vertex) {
        topologyPayload.add(vertex);

        for (Map.Entry<String, NumberTypedValue> entry : vertex.getAttributeToTypedValue().entrySet()) {
            String attribute = entry.getKey();
            DataType dataType = entry.getValue().getDataType();
            String vertexKey = vertex.getDefaultKey();

            dictionaryPayload.add(vertexKey, DictionaryPayload.Type.VERTEX, attribute, dataType);
        }
    }

    public void add(Edge edge) {
        topologyPayload.add(edge);
    }

    public void add(Event event) {
        eventsPayload.add(event);
    }

    public void add(Sample sample) {
        timeseriesPayload.add(sample);
        dictionaryPayload.add(
                sample.getVertexKey(),
                DictionaryPayload.Type.TIMESERIES,
                sample.getCounter(),
                sample.getDataType());
    }

    ////////////////////////////////////////////////////////////
    // Create class per OldPayload Type
    ////////////////////////////////////////////////////////////

    private static class EventsPayload {

        private List<Event> events = new LinkedList<>();

        private JSONObject doFillJson(JSONObject jsonToFill, String collectorId) {
            JSONArray eventsJsonArray = new JSONArray();
            for (Event event: events) {
                eventsJsonArray.put(event.toJson(collectorId));
            }

            jsonToFill.put("events", eventsJsonArray);

            return jsonToFill;
        }

        public void add(Event event) {
            events.add(event);
        }

    }

    private static class TopologyPayload  {

        private List<Vertex> vertices = new LinkedList<>();
        private List<Edge> edges = new LinkedList<>();


        private JSONObject doFillJson(JSONObject jsonToFill) {
            JSONArray verticesJsonArray = new JSONArray();
            for (Vertex vertex : vertices) {
                verticesJsonArray.put(vertex.toJson());
            }
            JSONArray edgesJsonArray = new JSONArray();
            for (Edge edge : edges) {
                edgesJsonArray.put(edge.toJson());
            }


            jsonToFill.put("vertices", verticesJsonArray);
            jsonToFill.put("edges", edgesJsonArray);

            return jsonToFill;
        }

        public void add(Vertex vertex) {
            vertices.add(vertex);
        }

        public void add(Edge edge) {
            edges.add(edge);
        }
    }

    private static class TimeseriesPayload {

        private List<Sample> samples = new LinkedList<>();

        private void doFillJson(JSONObject jsonToFill) {
            JSONObject jsonObject = new JSONObject();
            for (Sample sample : samples) {
                String vertexKey = sample.getVertexKey();
                String counter = sample.getCounter();
                Number value = sample.getValue();

                // ITculate json format is double in Sec
                double timestamp = sample.getTimestamp() / 1000;

                JSONObject counterValues = jsonObject.optJSONObject(vertexKey);
                if (counterValues == null) {
                    counterValues = new JSONObject();
                    jsonObject.put(vertexKey, counterValues);
                }

                JSONArray values = counterValues.optJSONArray(counter);
                if (values == null)
                    values = new JSONArray();
                counterValues.put(counter, values);

                JSONArray timestampValueTuple = new JSONArray();
                timestampValueTuple.put(timestamp);
                timestampValueTuple.put(value);
                values.put(timestampValueTuple);
            }

            jsonToFill.put("samples", jsonObject);
        }

        public void add(Sample sample) {
            samples.add(sample);
        }
    }

    private static class DictionaryPayload  {

        private Map<String, Map<DictionaryPayload.Type, Map<String, JSONObject>>> dictionary = new HashMap<>();

        private boolean changed = false;

        public boolean changed() {
            return changed;
        }

        private void doFillJson(JSONObject jsonToFill) {
            JSONObject jsonObject = new JSONObject();
            for (Map.Entry<String, Map<DictionaryPayload.Type, Map<String, JSONObject>>> vertexKeyEntry : dictionary.entrySet()) {
                String vertexKey = vertexKeyEntry.getKey();
                JSONObject vertexKeyJosnObject = jsonObject.optJSONObject(vertexKey);
                if (vertexKeyJosnObject == null) {
                    vertexKeyJosnObject = new JSONObject();
                    jsonObject.put(vertexKey, vertexKeyJosnObject);
                }

                for (Map.Entry<DictionaryPayload.Type, Map<String, JSONObject>> typeEntry : vertexKeyEntry.getValue().entrySet()) {
                    DictionaryPayload.Type type = typeEntry.getKey();
                    JSONObject typeJsonObject = vertexKeyJosnObject.optJSONObject(type.getValue());
                    if (typeJsonObject == null) {
                        typeJsonObject = new JSONObject();
                        vertexKeyJosnObject.put(type.getValue(), typeJsonObject);
                    }


                    for (Map.Entry<String, JSONObject> attributeEntry : typeEntry.getValue().entrySet()) {
                        String attribute = attributeEntry.getKey();
                        JSONObject metaData = attributeEntry.getValue();
                        typeJsonObject.put(attribute, metaData);
                    }
                }
            }


            jsonToFill.put("dictionary", jsonObject);
        }

        public boolean add(String vertexKey, DictionaryPayload.Type dictionaryType, String attribute, DataType dataType) {

            Map<DictionaryPayload.Type, Map<String, JSONObject>> vertexTypeJsonObject = dictionary.get(vertexKey);
            if (vertexTypeJsonObject == null) {
                vertexTypeJsonObject = new HashMap<>();
                dictionary.put(vertexKey, vertexTypeJsonObject);
            }

            Map<String, JSONObject> dictionaryTypeJsonObject = vertexTypeJsonObject.get(dictionaryType);
            if (dictionaryTypeJsonObject == null) {
                dictionaryTypeJsonObject = new HashMap<>();
                vertexTypeJsonObject.put(dictionaryType, dictionaryTypeJsonObject);
            }

            JSONObject currentMetaData = dataType.toJson();

            JSONObject dictionaryMetaData = dictionaryTypeJsonObject.get(attribute);
            if (dictionaryMetaData == null) {
                dictionaryTypeJsonObject.put(attribute, currentMetaData);
                changed = true;
                return true;
            } else if (!compareJson(dictionaryMetaData, currentMetaData)) {
                // add replace
                dictionaryTypeJsonObject.put(attribute, currentMetaData);
                changed = true;
                return true;
            }
            return false;
        }

        public static enum Type {
            VERTEX("vertex"),
            TIMESERIES("timeseries");

            String value;

            Type(String value) {
                this.value = value;
            }

            public String getValue() {
                return value;
            }
        }

        private boolean compareJson(JSONObject json1, JSONObject json2) {

            if (json1 == null && json2 == null)
                return true;

            if (json1 == null || json2 == null)
                return false;

            if (json1.length() != json2.length())
                return false;

            Set<String> json1keySet = dictionary.keySet();

            for (String json1key : json1keySet) {
                Object value1 = json1.opt(json1key);
                Object value2 = json2.opt(json1key);
                if (value1 == null && value2 == null)
                    continue;

                if (value1 == null || value2 == null)
                    return false;

                if (!value1.getClass().equals(value2.getClass()))
                    return false;

                if (value1 instanceof JSONObject) {
                    if (!this.compareJson((JSONObject) value1, (JSONObject) value2))
                        return false;
                }
                else if (value1 instanceof JSONArray) {
                    if (!this.compareJson((JSONArray) value1, (JSONArray) value2))
                        return false;
                }
                else {
                    if (!this.compareJson(value1, value2))
                        return false;
                }
            }

            return true;

        }

        private boolean compareJson(JSONArray json1, JSONArray json2) {

            if (json1 == null && json2 == null)
                return true;

            if (json1 == null || json2 == null)
                return false;

            if (json1.length() != json2.length())
                return false;

            for (int i = 0; i < json1.length(); i++) {
                Object value1 = json1.opt(i);
                Object value2 = json2.opt(i);

                if (value1 == null || value2 == null)
                    return false;

                if (!value1.getClass().equals(value2.getClass()))
                    return false;

                if (value1 instanceof JSONObject) {
                    if (!this.compareJson((JSONObject) value1, (JSONObject) value2))
                        return false;
                }
                else if (value1 instanceof JSONArray) {
                    if (!this.compareJson((JSONArray) value1, (JSONArray) value2))
                        return false;
                }
                else {
                    if (!this.compareJson(value1, value2))
                        return false;
                }
            }

            return true;

        }

        private boolean compareJson(Object value1, Object value2) {
            if (value1 == null && value2 == null)
                return true;

            if (value1 == null || value2 == null)
                return false;

            return value1.equals(value2);
        }

    }


}
