//
// (C) ITculate, Inc. 2015-2017
// All rights reserved
// Licensed under MIT License (see LICENSE)
//

package com.itculate.sdk;


import com.itculate.sdk.provider.AgentProvider;
import com.itculate.sdk.provider.Provider;
import com.itculate.sdk.provider.SynchronousApiProvider;


import java.util.*;

public class ITculateSDK {


    private static String _DEFAULT_COLLECTOR_ID = "sdk";

    private Provider provider;

    private ITculateSDK(Provider provider) {
        this.provider = provider;
    }

    static public Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private String serverUrl = null;
        private boolean agent = true;
        private String apiKey;
        private String apiSecret;
        private Provider provider;

        public Builder serverUrl(String serverUrl) {
            this.serverUrl = serverUrl;
            return this;
        }

        public Builder agent(boolean agent) {
            this.agent = agent;
            return this;
        }

        public Builder provider(Provider provider) {
            this.provider = provider;
            this.agent = false;
            return this;
        }

        public Builder apiKey(String apiKey) {
            this.apiKey = apiKey;
            return this;
        }

        public Builder apiSecret(String apiSecret) {
            this.apiSecret = apiSecret;
            return this;
        }

        public ITculateSDK build() {
            if (provider != null) {
                if (agent)
                    throw new IllegalStateException("When use provider, agent cannot be set");
                if (serverUrl != null)
                    throw new IllegalStateException("When use provider, serverUrl cannot be set");
                if (apiKey != null || apiSecret != null)
                    throw new IllegalStateException("When use provider, apiKey and apiSecret cannot be set");
            }
            else {
                if (agent) {
                    provider = new AgentProvider(serverUrl);
                } else {
                    provider = new SynchronousApiProvider(serverUrl, apiKey, apiSecret);
                }
            }
            return new ITculateSDK(provider);
        }
    }

    private Map<String, Payload> collectorIdToPayloads = new HashMap<String, Payload>();

    public void addVertex(String collectorId, Vertex vertex) throws ITculateApiException {
        getPayload(collectorId).add(vertex);
    }

    public void addEdge(String collectorId, Edge edge) {
        getPayload(collectorId).add(edge);
    }

    public void addEvent(String collectorId, Event event) {
        getPayload(collectorId).add(event);
    }

    public void addSample(Sample sample) {
        getPayload(_DEFAULT_COLLECTOR_ID).add(sample);
    }

    public void flushAll() throws ITculateApiException {
        for (Payload payload : collectorIdToPayloads.values()) {
            provider.flush(payload.getCollectorId(), payload.toJson());
        }
    }

    private Payload getPayload(String collectorId) {
        Payload payload = collectorIdToPayloads.get(collectorId);
        if (payload == null)
            payload = new Payload(collectorId);
        collectorIdToPayloads.put(collectorId, payload);
        return payload;
    }
}
