//
// (C) ITculate, Inc. 2015-2017
// All rights reserved
// Licensed under MIT License (see LICENSE)
//

package com.itculate.sdk;


import org.json.JSONObject;

public class Event {

    private final String vertexKey;
    private final String eventType;
    private final String message;
    private final String severity;
    private final long timestamp;

    private Event(Builder builder) {
        this.vertexKey = builder.vertexKey;
        this.eventType = builder.eventType;
        this.message = builder.message;
        this.severity = builder.severity;
        this.timestamp = builder.timestamp;
    }

    public String getVertexKey() {
        return vertexKey;
    }

    public String getMessage() {
        return message;
    }

    public String getSeverity() {
        return severity;
    }

    public long getTimestamp() {
        return timestamp;
    }

    public String getEventType() {
        return eventType;
    }

    static public Event.Builder builder() {
        return new Event.Builder();
    }

    static public Event.HealthyBuilder healthyBuilder() {
        return new Event.HealthyBuilder();
    }

    static public Event.UnhealthyBuilder unhealthyBuilder() {
        return new Event.UnhealthyBuilder();
    }

    public JSONObject toJson(String collectorId) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("collector_id", collectorId);
        jsonObject.put("vertex_key", this.vertexKey);
        jsonObject.put("event_time", this.timestamp / 1000);
        jsonObject.put("event_type", this.eventType);
        jsonObject.put("severity", this.severity);
        jsonObject.put("message", this.message);

        return jsonObject;
    }

    public static class Builder {
        String vertexKey;
        String eventType;
        String message;
        String severity;
        long timestamp = 0;

        public Builder vertexKey(String vertexKey) {
            this.vertexKey = vertexKey;
            return this;
        }

        public Builder vertexKey(Vertex vertex) {
            this.vertexKey = vertex.getDefaultKey();
            return this;
        }

        public Builder eventType(String eventType) {
            this.eventType = eventType;
            return this;
        }

        public Builder message(String message) {
            this.message = message;
            return this;
        }

        public Builder severity(String severity) {
            this.severity = severity;
            return this;
        }

        public Builder timestamp(long timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public Event build() {

            if (this.vertexKey == null)
                throw new IllegalStateException("vertexKey must be set");

            if (this.eventType == null)
                throw new IllegalStateException("eventType must be set");

            if (this.severity == null)
                throw new IllegalStateException("severity must be set");

            if (this.message == null)
                throw new IllegalStateException("message must be set");

            if (this.timestamp == 0)
                this.timestamp = System.currentTimeMillis();

            return new Event(this);
        }
    }

    public static class HealthyBuilder {

        String vertexKey;
        String message;
        long timestamp = 0;

        public HealthyBuilder vertexKey(String vertexKey) {
            this.vertexKey = vertexKey;
            return this;
        }

        public HealthyBuilder vertexKey(Vertex vertex) {
            this.vertexKey = vertex.getDefaultKey();
            return this;
        }

        public HealthyBuilder message(String message) {
            this.message = message;
            return this;
        }

        public HealthyBuilder timestamp(long timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public Event build() {
            return new Event.Builder().
                    vertexKey(this.vertexKey).
                    message(this.message).
                    severity("SUCCESS").
                    eventType("HEALTHY").
                    timestamp(this.timestamp).build();
        }
    }

    public static class UnhealthyBuilder {

        String vertexKey;
        String message;
        long timestamp = 0;

        public UnhealthyBuilder vertexKey(String vertexKey) {
            this.vertexKey = vertexKey;
            return this;
        }

        public UnhealthyBuilder vertexKey(Vertex vertex) {
            this.vertexKey = vertex.getDefaultKey();
            return this;
        }

        public UnhealthyBuilder message(String message) {
            this.message = message;
            return this;
        }

        public UnhealthyBuilder timestamp(long timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public Event build() {
            return new Event.Builder().
                    vertexKey(this.vertexKey).
                    message(this.message).
                    severity("ERROR").
                    eventType("UNHEALTHY").
                    timestamp(this.timestamp).build();
        }
    }
}
