//
// (C) ITculate, Inc. 2015-2017
// All rights reserved
// Licensed under MIT License (see LICENSE)
//

package com.itculate.sdk;

import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;

public class Edge {

    private Map<String, String> source;
    private Map<String, String> target;
    private String topology;

    private Edge(Builder builder) {
        this.source = builder.source;
        this.target = builder.target;
        this.topology = builder.topology;
    }

    static public Edge.Builder builder() {
        return new Edge.Builder();
    }

    public static class Builder {
        private Map<String, String> source;
        private Map<String, String> target;
        private String topology;

        public Builder source(Map<String, String> source) {
            this.source = source;
            return this;
        }

        public Builder source(String sourceKey) {
            this.source = new HashMap<>();
            this.source.put("pk", sourceKey);
            return this;
        }

        public Builder source(Vertex vertex) {
            this.source = vertex.getKeys();
            return this;
        }

        public Builder target(Map<String, String> keys) {
            this.target = keys;
            return this;
        }

        public Builder target(String targetKey) {
            this.target = new HashMap<>();
            this.target.put("pk", targetKey);
            return this;
        }

        public Builder target(Vertex vertex) {
            this.target = vertex.getKeys();
            return this;
        }

        public Builder topology(String type) {
            this.topology = type;
            return this;
        }

        public Edge build() {
            if (this.topology == null || this.source == null || this.target == null)
                throw new IllegalStateException("getTypedValue, source and target must be provided");

            if (this.source.isEmpty())
                throw new IllegalStateException("source keys cannot be empty");

            if (this.target.isEmpty())
                throw new IllegalStateException("target keys cannot be empty");

            return new Edge(this);
        }
    }

    public JSONObject toJson() {

        JSONObject jsonObject = new JSONObject();
        jsonObject.put("_type", this.topology);

        JSONObject sourceKeysJsonObject = new JSONObject();
        for (Map.Entry<String, String> next : source.entrySet()) {
            sourceKeysJsonObject.put(next.getKey(), next.getValue());
        }
        JSONObject targetKeysJsonObject = new JSONObject();
        for (Map.Entry<String, String> next : target.entrySet()) {
            targetKeysJsonObject.put(next.getKey(), next.getValue());
        }

        jsonObject.put("_source_keys", sourceKeysJsonObject);
        jsonObject.put("_target_keys", targetKeysJsonObject);
        return jsonObject;
    }
}
