//
// (C) ITculate, Inc. 2015-2017
// All rights reserved
// Licensed under MIT License (see LICENSE)
//

package com.itculate.sdk.provider;

import com.itculate.sdk.ITculateApiException;
import org.apache.http.HttpHost;
import org.apache.http.client.AuthCache;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.entity.AbstractHttpEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.impl.client.BasicAuthCache;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.json.JSONObject;

import java.io.IOException;
import java.net.InetAddress;
import java.net.URI;


public class AgentProvider implements Provider {

    private static String DEFAULT_AGENT_REST_URL = "http://localhost:8000";

    private String serverUrl = null;

    public AgentProvider(String serverUrl) {
        if (serverUrl == null)
            serverUrl = DEFAULT_AGENT_REST_URL;

        this.serverUrl = serverUrl;
    }

    /**
     * Submits a custom ITculate event via the ITculate Agent.
     *
     * @throws ITculateApiException if failed.
     */
    public void flush(String collectorId, JSONObject jsonObject) throws ITculateApiException {

        if (serverUrl == null) {
            throw new IllegalStateException("serverUrl can not be null");
        }

        if (jsonObject == null) {
            throw new IllegalStateException("jsonObject must not be null");
        }

        if (collectorId == null) {
            throw new IllegalStateException("collectorId must not be null");
        }

        URI uri = URI.create(this.serverUrl);
        String host = uri.getHost();
        int port = uri.getPort();
        String scheme = uri.getScheme();


        HttpHost targetHost = new HttpHost(host, port, scheme);

        // Create AuthCache instance
        AuthCache authCache = new BasicAuthCache();
        // Generate BASIC scheme object and add it to the local auth cache
        BasicScheme basicAuth = new BasicScheme();
        authCache.put(targetHost, basicAuth);


        // Add AuthCache to the execution context
        HttpClientContext context = HttpClientContext.create();
        context.setAuthCache(authCache);

        RequestConfig globalConfig = RequestConfig.custom().setCookieSpec(CookieSpecs.DEFAULT).build();
        CloseableHttpClient httpclient = HttpClients.custom().setDefaultRequestConfig(globalConfig).build();

        try {
            jsonObject.put("host", InetAddress.getLocalHost().getHostAddress());
            // add Property override the values if exist
            jsonObject.put("collector_id", collectorId);
            jsonObject.put("collector_version", "0.1.1");

            HttpPost httpPost = new HttpPost("/api/v1/upload");
            AbstractHttpEntity entity = new StringEntity(jsonObject.toString());
            entity.setContentType("application/json");
            httpPost.setEntity(entity);

            CloseableHttpResponse response = httpclient.execute(targetHost, httpPost, context);
            Util.checkResponse(response);
        } catch (Exception e) {
            throw new ITculateApiException("Failed to flush collectorId " + collectorId, e);
        }
    }
}

