//
// (C) ITculate, Inc. 2015-2017
// All rights reserved
// Licensed under MIT License (see LICENSE)
//

package com.itculate.sdk.provider;

import com.itculate.sdk.ITculateApiException;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.json.JSONArray;
import org.json.JSONObject;
import org.msgpack.MessagePack;
import org.msgpack.packer.BufferPacker;

import javax.xml.bind.DatatypeConverter;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.zip.GZIPOutputStream;

public class Util {
    public static String compressForJson(JSONObject jsonObject) throws IOException {
        byte[] packed = Util.msgpack(jsonObject);
        byte[] compressed = Util.compress(packed);
        return hexlify(compressed);
    }

    static byte[] compress(byte[] bytes) throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        GZIPOutputStream gzip = new GZIPOutputStream(out);
        gzip.write(bytes);
        gzip.close();

        return out.toByteArray();
    }

    static String hexlify(byte[] bytes) {
        return DatatypeConverter.printHexBinary(bytes);
    }

    static byte[] msgpack(JSONObject json) throws IOException {
        BufferPacker bufferPacker = new MessagePack().createBufferPacker();
        try {
            JsonElementPacker packer = new JsonElementPacker(bufferPacker);
            return packer.pack(json);
        } finally {
            bufferPacker.close();
        }
    }

    static void checkResponse(CloseableHttpResponse response) throws ITculateApiException {
        int statusCode = response.getStatusLine().getStatusCode();
        if (statusCode != 201) {
            String reasonPhrase = response.getStatusLine().getReasonPhrase();
            throw new ITculateApiException("Http Error " + statusCode + " " + reasonPhrase +
                    " \r\n" + response.toString());
        }
    }

}


class JsonElementPacker {
    private BufferPacker packer;

    JsonElementPacker(BufferPacker packer) {
        this.packer = packer;
    }

    byte[] pack(JSONObject jsonObject) throws IOException {
        doPack(jsonObject);
        return packer.toByteArray();
    }

    private void doPack(JSONObject jsonObject) throws IOException {
        packer.writeMapBegin(jsonObject.length());
        for (Object key : jsonObject.keySet()) {
            packer.write(key);
            Object value = jsonObject.get((String) key);
            if (value instanceof JSONObject)
                doPack((JSONObject) value);
            else if (value instanceof JSONArray)
                doPack((JSONArray) value);
            else
                doPack(value);

        }
        packer.writeMapEnd();
    }

    private void doPack(JSONArray jsonArray) throws IOException {
        packer.writeArrayBegin(jsonArray.length());
        for (int i = 0; i < jsonArray.length(); i++) {
            Object value = jsonArray.get(i);
            if (value instanceof JSONObject)
                doPack((JSONObject) value);
            else if (value instanceof JSONArray)
                doPack((JSONArray) value);
            else
                doPack(value);
        }
        packer.writeArrayEnd();
    }

    private void doPack(Object value) throws IOException {
        packer.write(value);
    }

}