//
// (C) ITculate, Inc. 2015-2017
// All rights reserved
// Licensed under MIT License (see LICENSE)
//

package com.itculate.sdk.provider;

import com.itculate.sdk.ITculateApiException;
import org.apache.http.HttpHost;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.AuthCache;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.entity.AbstractHttpEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.impl.client.BasicAuthCache;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.json.JSONObject;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.InetAddress;
import java.net.URI;
import java.nio.file.Paths;
import java.util.Properties;

/**
 * A simple client for submitting topologies via the ITculate API.
 */
public class SynchronousApiProvider implements Provider {

    private String apiKey = null;
    private String apiSecret = null;
    private String serverUrl = null;
    private String apiEndPoint = null;

    public SynchronousApiProvider(String serverUrl, String apiKey, String apiSecret) {

        if (serverUrl == null)
            serverUrl = DEFAULT_SYNCHRONOUS_API_URL;

        this.apiKey = apiKey;
        this.apiSecret = apiSecret;
        this.serverUrl = serverUrl;
        this.apiEndPoint = API_END_POINT;

        if (this.apiKey == null && this.apiSecret == null)
            loadCredentialsFromFile();
    }

    private void loadCredentialsFromFile() {

        Properties props = new Properties();

        String path = Paths.get(System.getProperty("user.home"), ".itculate", "credentials").toString();

        File file = new File(path);
        if (!file.exists())
            throw new IllegalStateException("Failed api key, file " + path + " doesn't exist");

        FileInputStream input = null;
        try {
            input = new FileInputStream(file);
            props.load(input);
            this.apiKey = props.getProperty("api_key");
            if (this.apiKey == null)
                throw new IllegalStateException("Failed to find api_key in " + path);

            this.apiSecret = props.getProperty("api_secret");
            if (this.apiSecret == null)
                throw new IllegalStateException("Failed to find api_secret in " + path);

        } catch (IOException e) {
            throw new IllegalStateException("Failed to load api_key and api_secret from " + path);
        } finally {
            if (input != null)
                try {
                    input.close();
                } catch (IOException e) {
                    // do nothing
                }
        }
    }

    /**
     * Submits a custom ITculate event via the ITculate API.
     *
     * @return the response status returned by the ITculate API.
     * @throws ITculateApiException if failed.
     */
    public int flush(String collectorId, JSONObject jsonObject) throws ITculateApiException {

        if (this.serverUrl == null || this.apiKey == null || this.apiSecret == null) {
            throw new IllegalStateException("serverUrl, apiKey and apiSecret cannot be null");
        }

        if (jsonObject == null) {
            throw new IllegalStateException("jsonObject must not be null");
        }

        if (collectorId == null) {
            throw new IllegalStateException("collectorId must not be null");
        }

        try {
            JSONObject jsonEntity = new JSONObject();
            jsonEntity.put("compressed_payload", Util.compressForJson(jsonObject));
            jsonEntity.put("collector_id", collectorId);
            jsonEntity.put("collector_version", "0.1.1");
            jsonEntity.put("host", InetAddress.getLocalHost().getHostAddress());
            String payload = jsonEntity.toString();

            URI uri = URI.create(this.serverUrl);
            return doPost(uri, payload);
        } catch (IOException e) {
            throw new ITculateApiException("Failed to flush payload collectorId " + collectorId, e);
        }
    }

    protected int doPost(URI apiServerUri, String payload) throws IOException {
        String host = apiServerUri.getHost();
        int port = apiServerUri.getPort();
        String scheme = apiServerUri.getScheme();


        HttpHost targetHost = new HttpHost(host, port, scheme);
        CredentialsProvider credsProvider = new BasicCredentialsProvider();
        UsernamePasswordCredentials credentials = new UsernamePasswordCredentials(apiKey, apiSecret);
        AuthScope authScope = AuthScope.ANY;
        credsProvider.setCredentials(authScope, credentials);

        // Create AuthCache instance
        AuthCache authCache = new BasicAuthCache();
        // Generate BASIC scheme object and add it to the local auth cache
        BasicScheme basicAuth = new BasicScheme();
        authCache.put(targetHost, basicAuth);


        // Add AuthCache to the execution context
        HttpClientContext context = HttpClientContext.create();
        context.setCredentialsProvider(credsProvider);
        context.setAuthCache(authCache);

        RequestConfig globalConfig = RequestConfig.custom().setCookieSpec(CookieSpecs.DEFAULT).build();
        CloseableHttpClient httpclient = HttpClients.custom().setDefaultRequestConfig(globalConfig).build();

        AbstractHttpEntity entity = new StringEntity(payload);

        entity.setContentType("application/json");

        HttpPost httpPost = new HttpPost(apiEndPoint);
        httpPost.setEntity(entity);

        CloseableHttpResponse response = httpclient.execute(targetHost, httpPost, context);
        return response.getStatusLine().getStatusCode();
    }
}
