//
// (C) ITculate, Inc. 2015-2017
// All rights reserved
// Licensed under MIT License (see LICENSE)
//

package com.itculate.sdk.payloads;

import com.itculate.sdk.types.DataType;
import org.json.JSONArray;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;


public class DictionaryPayload extends Payload {


    private Map<String, Map<Type, Map<String, JSONObject>>> dictionary = new HashMap<>();

    private boolean changed = false;

    public DictionaryPayload(String collectorId) {
        super(collectorId);
    }

    @Override
    public boolean changed() {
        return changed;
    }

    @Override
    final public JSONObject toJson(boolean reset) {
        Map<String, Map<Type, Map<String, JSONObject>>> localDictionary;
        synchronized (this) {
            localDictionary = dictionary;
            if (reset) {
                dictionary = new HashMap<>();
                changed = false;
            }
        }

        JSONObject jsonObject = new JSONObject();
        doFillJson(localDictionary, jsonObject);
        return jsonObject;
    }

    private void doFillJson(Map<String, Map<Type, Map<String, JSONObject>>> localDictionary, JSONObject jsonToFill) {
        JSONObject jsonObject = new JSONObject();
        for (Map.Entry<String, Map<Type, Map<String, JSONObject>>> vertexKeydEntry : localDictionary.entrySet()) {
            String vertexKey = vertexKeydEntry.getKey();
            JSONObject vertexKeyJosnObject = jsonObject.optJSONObject(vertexKey);
            if (vertexKeyJosnObject == null) {
                vertexKeyJosnObject = new JSONObject();
                jsonObject.put(vertexKey, vertexKeyJosnObject);
            }

            for (Map.Entry<Type, Map<String, JSONObject>> typeEntry : vertexKeydEntry.getValue().entrySet()) {
                Type type = typeEntry.getKey();
                JSONObject typeJsonObject = vertexKeyJosnObject.optJSONObject(type.getValue());
                if (typeJsonObject == null) {
                    typeJsonObject = new JSONObject();
                    vertexKeyJosnObject.put(type.getValue(), typeJsonObject);
                }


                for (Map.Entry<String, JSONObject> attributeEntry : typeEntry.getValue().entrySet()) {
                    String attribute = attributeEntry.getKey();
                    JSONObject metaData = attributeEntry.getValue();
                    typeJsonObject.put(attribute, metaData);
                }
            }
        }


        jsonToFill.put("dictionary", jsonObject);
    }

    public boolean add(String vertexKey, Type dictionaryType, String attribute, DataType dataType) {

        Map<Type, Map<String, JSONObject>> vertexTypeJsonObject = dictionary.get(vertexKey);
        if (vertexTypeJsonObject == null) {
            vertexTypeJsonObject = new HashMap<>();
            dictionary.put(vertexKey, vertexTypeJsonObject);
        }

        Map<String, JSONObject> dictionaryTypeJsonObject = vertexTypeJsonObject.get(dictionaryType);
        if (dictionaryTypeJsonObject == null) {
            dictionaryTypeJsonObject = new HashMap<>();
            vertexTypeJsonObject.put(dictionaryType, dictionaryTypeJsonObject);
        }

        JSONObject currentMetaData = dataType.toJson();

        JSONObject dictionaryMetaData = dictionaryTypeJsonObject.get(attribute);
        if (dictionaryMetaData == null) {
            dictionaryTypeJsonObject.put(attribute, currentMetaData);
            changed = true;
            return true;
        } else if (!compareJson(dictionaryMetaData, currentMetaData)) {
            // add replace
            dictionaryTypeJsonObject.put(attribute, currentMetaData);
            changed = true;
            return true;
        }
        return false;
    }

    public enum Type {
        VERTEX("vertex"),
        TIMESERIES("timeseries");

        String value;

        Type(String value) {
            this.value = value;
        }

        public String getValue() {
            return value;
        }
    }

    private boolean compareJson(JSONObject json1, JSONObject json2) {

        if (json1 == null && json2 == null)
            return true;

        if (json1 == null || json2 == null)
            return false;

        if (json1.length() != json2.length())
            return false;

        Set<String> json1keySet = dictionary.keySet();

        for (String json1key : json1keySet) {
            Object value1 = json1.opt(json1key);
            Object value2 = json2.opt(json1key);
            if (value1 == null && value2 == null)
                continue;

            if (value1 == null || value2 == null)
                return false;

            if (!value1.getClass().equals(value2.getClass()))
                return false;

            if (value1 instanceof JSONObject) {
                if (!this.compareJson((JSONObject) value1, (JSONObject) value2))
                    return false;
            }
            else if (value1 instanceof JSONArray) {
                if (!this.compareJson((JSONArray) value1, (JSONArray) value2))
                    return false;
            }
            else {
                if (!this.compareJson(value1, value2))
                    return false;
            }
        }

        return true;

    }

    private boolean compareJson(JSONArray json1, JSONArray json2) {

        if (json1 == null && json2 == null)
            return true;

        if (json1 == null || json2 == null)
            return false;

        if (json1.length() != json2.length())
            return false;

        for (int i = 0; i < json1.length(); i++) {
            Object value1 = json1.opt(i);
            Object value2 = json2.opt(i);

            if (value1 == null || value2 == null)
                return false;

            if (!value1.getClass().equals(value2.getClass()))
                return false;

            if (value1 instanceof JSONObject) {
                if (!this.compareJson((JSONObject) value1, (JSONObject) value2))
                    return false;
            }
            else if (value1 instanceof JSONArray) {
                if (!this.compareJson((JSONArray) value1, (JSONArray) value2))
                    return false;
            }
            else {
                if (!this.compareJson(value1, value2))
                    return false;
            }
        }

        return true;

    }

    private boolean compareJson(Object value1, Object value2) {
        if (value1 == null && value2 == null)
            return true;

        if (value1 == null || value2 == null)
            return false;

        return value1.equals(value2);
    }

}
