//
// (C) ITculate, Inc. 2015-2017
// All rights reserved
// Licensed under MIT License (see LICENSE)
//

package com.itculate.sdk;

import com.itculate.sdk.types.NumberTypedValue;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;

public class Vertex {

    private String type;
    private String name;
    private Map<String, String> keys;
    private JSONObject data;

    private Map<String, NumberTypedValue> attributeToTypedValue;

    private Vertex(Builder builder) {
        this.type = builder.type;
        this.name = builder.name;
        this.keys = builder.keys;
        this.data = builder.data;

        attributeToTypedValue = new HashMap<>();
    }

    public String getType() {
        return type;
    }

    public String getName() {
        return name;
    }

    public Map<String, String> getKeys() {
        return keys;
    }

    public JSONObject getData() {
        return data;
    }

    boolean hasNumberTypedValue() {
        return getAttributeToTypedValue().size() > 0;
    }

    Map<String, NumberTypedValue> getAttributeToTypedValue() {
        // used internal for populating dictionary
        return attributeToTypedValue;
    }

    public void addProperty(String property, NumberTypedValue value) {
        this.data.put(property, value.getValue());

        attributeToTypedValue.put(property, value);
    }

    public void addProperty(String property, String value) {
        this.data.put(property, value);
    }

    public void addProperty(String property, boolean value) {
        this.data.put(property, value);
    }

    public String getDefaultKey() {
        return keys.values().iterator().next();
    }

    static public Vertex.Builder builder() {
        return new Vertex.Builder();
    }

    public static class Builder {
        private String type;
        private String name;
        private Map<String, String> keys;
        private JSONObject data;

        public Builder type(String type) {
            this.type = type;
            return this;
        }

        public Builder name(String name) {
            this.name = name;
            return this;
        }

        public Builder keys(Map<String, String> keys) {
            this.keys = keys;
            return this;
        }

        public Builder keys(String key) {
            this.keys = new HashMap<>();
            this.keys.put("pk", key);
            return this;
        }

        public Builder data(JSONObject data) {
            this.data = data;
            return this;
        }

        public Vertex build() {
            if (this.type == null || this.name == null || this.keys == null)
                throw new IllegalStateException("getTypedValue, name and keys must be provided");

            if (this.keys.isEmpty())
                throw new IllegalStateException("keys cannot be empty");

            if (this.data == null)
                this.data = new JSONObject();

            return new Vertex(this);
        }
    }

    public JSONObject toJson() {

        JSONObject jsonObject = new JSONObject();
        jsonObject.put("_type", this.type);
        jsonObject.put("_name", this.name);

        JSONObject jsonKeys = new JSONObject();
        for (Map.Entry<String, String> entry : keys.entrySet()) {
            jsonKeys.put(entry.getKey(), entry.getValue());
        }

        jsonObject.put("_keys", jsonKeys);

        if (this.data != null) {
            for (Object key : this.data.keySet()) {
                String skey = (String) key;
                jsonObject.put(skey, this.data.get(skey));
            }
        }
        return jsonObject;
    }
}
