//
// (C) ITculate, Inc. 2015-2017
// All rights reserved
// Licensed under MIT License (see LICENSE)
//

package com.itculate.sdk;


import com.itculate.sdk.payloads.*;
import com.itculate.sdk.payloads.DictionaryPayload;
import com.itculate.sdk.provider.AgentProvider;
import com.itculate.sdk.provider.Provider;
import com.itculate.sdk.provider.SynchronousApiProvider;
import com.itculate.sdk.types.DataType;
import com.itculate.sdk.types.NumberTypedValue;


import java.util.*;

public class ITculateSDK {

    private static String _DEFAULT_COLLECTOR_ID = "sdk";

    private Provider provider;

    private ITculateSDK(Provider provider) {
        this.provider = provider;
    }

    static public Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private String serverUrl = null;
        private boolean agent = true;
        private String apiKey;
        private String apiSecret;
        private Provider provider;

        public Builder serverUrl(String serverUrl) {
            this.serverUrl = serverUrl;
            return this;
        }

        public Builder agent(boolean agent) {
            this.agent = agent;
            return this;
        }

        public Builder provider(Provider provider) {
            this.provider = provider;
            this.agent = false;
            return this;
        }

        public Builder apiKey(String apiKey) {
            this.apiKey = apiKey;
            return this;
        }

        public Builder apiSecret(String apiSecret) {
            this.apiSecret = apiSecret;
            return this;
        }

        public ITculateSDK build() {
            if (provider != null) {
                if (agent)
                    throw new IllegalStateException("When use provider, agent cannot be set");
                if (serverUrl != null)
                    throw new IllegalStateException("When use provider, serverUrl cannot be set");
                if (apiKey != null || apiSecret != null)
                    throw new IllegalStateException("When use provider, apiKey and apiSecret cannot be set");
            }
            else {
                if (agent) {
                    provider = new AgentProvider(serverUrl);
                } else {
                    provider = new SynchronousApiProvider(serverUrl, apiKey, apiSecret);
                }
            }
            return new ITculateSDK(provider);
        }
    }

    public void addVertex(String collectorId, Vertex vertex) throws ITculateApiException {
        TopologyPayload topologyPayload = topologyPayloads.getPayload(collectorId);
        topologyPayload.add(vertex);

        DictionaryPayload dictionaryPayload = dictionaryPayloads.getPayload(_DEFAULT_COLLECTOR_ID);
        for (Map.Entry<String, NumberTypedValue> entry : vertex.getAttributeToTypedValue().entrySet()) {
            String attribute = entry.getKey();
            DataType dataType = entry.getValue().getDataType();
            String vertexKey = vertex.getDefaultKey();

            dictionaryPayload.add(vertexKey, DictionaryPayload.Type.VERTEX, attribute, dataType);
        }
    }

    public void addEdge(String collectorId, Edge edge) {
        TopologyPayload topologyPayload = topologyPayloads.getPayload(collectorId);
        topologyPayload.add(edge);
    }

    public void addEvent(Event event) {
        EventsPayload eventsPayload = eventsPayloads.getPayload(_DEFAULT_COLLECTOR_ID);
        eventsPayload.add(event);
    }

    public void addSample(Sample sample) {
        TimeseriesPayload timeseriesPayload = timeseriesPayloads.getPayload(_DEFAULT_COLLECTOR_ID);
        timeseriesPayload.add(sample);

        if (sample.getDataType() != null) {
            DictionaryPayload dictionaryPayload = dictionaryPayloads.getPayload(_DEFAULT_COLLECTOR_ID);
            dictionaryPayload.add(
                    sample.getVertexKey(),
                    DictionaryPayload.Type.TIMESERIES,
                    sample.getCounter(),
                    sample.getDataType());
        }
    }

    public void flushAll() throws ITculateApiException {
        topologyPayloads.flush(true);
        timeseriesPayloads.flush(true);
        dictionaryPayloads.flush(false);
        eventsPayloads.flush(true);
    }

    public void print() {
        topologyPayloads.print();
        timeseriesPayloads.print();
        dictionaryPayloads.print();
    }

    private CollectorsPayloads<TopologyPayload> topologyPayloads = new CollectorsPayloads<TopologyPayload>() {
        @Override
        public TopologyPayload createPayload(String createPayload) {
            return new TopologyPayload(createPayload);
        }
    };

    private CollectorsPayloads<TimeseriesPayload> timeseriesPayloads = new CollectorsPayloads<TimeseriesPayload>() {
        @Override
        public TimeseriesPayload createPayload(String createPayload) {
            return new TimeseriesPayload(createPayload);
        }
    };


    private CollectorsPayloads<DictionaryPayload> dictionaryPayloads = new CollectorsPayloads<DictionaryPayload>() {
        @Override
        public DictionaryPayload createPayload(String createPayload) {
            return new DictionaryPayload(createPayload);
        }
    };

    private CollectorsPayloads<EventsPayload> eventsPayloads = new CollectorsPayloads<EventsPayload>() {
        @Override
        public EventsPayload createPayload(String createPayload) {
            return new EventsPayload(createPayload);
        }
    };


    abstract class CollectorsPayloads<T extends Payload> {

        Map<String, T> collectorIdToPayloads = new HashMap<>();

        private T getPayload(String collectorId) {
            T payload = collectorIdToPayloads.get(collectorId);
            if (payload == null)
                payload = createPayload(collectorId);
            collectorIdToPayloads.put(collectorId, payload);
            return payload;
        }

        private void print() {
            for (T payload : collectorIdToPayloads.values()) {
                System.out.println(payload.getClass().getSimpleName() + " collectorId: " + payload.getCollectorId());
            }
        }

        private void flush(boolean reset) throws ITculateApiException {
            for (T payload : collectorIdToPayloads.values()) {
                String className = payload.getClass().getSimpleName();
                String collectorId = payload.getCollectorId();

                if (payload.changed()) {
                    provider.flush(payload.getCollectorId(), payload.toJson(reset));
                } else {
                    System.out.println("Flush skipped, no change " + className + " collectorId: " + collectorId);
                }
            }
        }

        abstract protected T createPayload(String collectorId);
    }
}
