package io.intercom.android.nexus;

import androidx.annotation.Nullable;
import android.text.TextUtils;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import static io.intercom.android.nexus.NexusEventType.AdminIsTyping;
import static io.intercom.android.nexus.NexusEventType.AdminIsTypingANote;
import static io.intercom.android.nexus.NexusEventType.ConversationAssigned;
import static io.intercom.android.nexus.NexusEventType.ConversationClosed;
import static io.intercom.android.nexus.NexusEventType.ConversationReopened;
import static io.intercom.android.nexus.NexusEventType.ConversationSeen;
import static io.intercom.android.nexus.NexusEventType.NewComment;
import static io.intercom.android.nexus.NexusEventType.NewNote;
import static io.intercom.android.nexus.NexusEventType.Subscribe;
import static io.intercom.android.nexus.NexusEventType.Unsubscribe;
import static io.intercom.android.nexus.NexusEventType.UserContentSeenByAdmin;
import static io.intercom.android.nexus.NexusEventType.UserIsTyping;
import static io.intercom.android.nexus.NexusEventType.UserPresence;

public class NexusEvent {

    static final String TOPIC_PREFIX_CONVERSATION = "conversation/";
    static final String EVENT_GUID = "eventGuid";
    static final String EVENT_DATA = "eventData";
    static final String EVENT_NAME = "eventName";
    static final String NX_TOPICS = "nx.Topics";
    static final String NX_TO_USER = "nx.ToUser";
    static final String NX_FROM_USER = "nx.FromUser";

    static final String ADMIN_ID = "adminId";
    static final String ADMIN_NAME = "adminName";
    static final String ADMIN_AVATAR = "adminAvatar";
    static final String ADMIN_TIMESTAMP = "adminTimestamp";
    static final String CONVERSATION_ID = "conversationId";
    static final String ASSIGNEE_ID = "assigneeId";

    private final NexusEventType eventType;
    private final EventData eventData;
    private final List<String> topics;
    private final String guid;
    private final String userId;

    public NexusEvent(Builder builder) {
        eventType = builder.eventName;
        eventData = new EventData();
        if (builder.eventData != null) {
            Set<String> keys = builder.eventData.keySet();
            for (String key : keys) {
                Object object = builder.eventData.get(key);
                if (object != null) {
                    this.eventData.put(key, object);
                }
            }
        }

        topics = new ArrayList<>();
        if (builder.topics != null) {
            for (String topic : builder.topics) {
                if (!TextUtils.isEmpty(topic)) {
                    topics.add(topic);
                }
            }
        }
        guid = UUID.randomUUID().toString();
        userId = builder.userId != null ? builder.userId : "";
    }

    public NexusEvent(JSONObject jsonObject) {
        eventType = NexusEventType.safeValueOf(jsonObject.optString(EVENT_NAME));

        JSONObject eventDataJson = jsonObject.optJSONObject(EVENT_DATA);

        if (eventDataJson != null && eventDataJson.length() > 0) {
            eventData = new EventData(eventDataJson.length());
            Iterator<String> keys = eventDataJson.keys();
            while (keys.hasNext()) {
                String key = keys.next();
                Object value = eventDataJson.opt(key);
                if (value != null) {
                    eventData.put(key, value);
                }
            }
        } else {
            eventData = new EventData();
        }

        topics = new ArrayList<>();
        JSONArray topicsArray = jsonObject.optJSONArray(NX_TOPICS);
        if (topicsArray != null) {
            for (int i = 0; i < topicsArray.length(); i++) {
                String topic = topicsArray.optString(i);
                if (!TextUtils.isEmpty(topic)) {
                    topics.add(topic);
                }
            }
        }

        guid = jsonObject.optString(EVENT_GUID);

        if (jsonObject.has(NX_TO_USER)) {
            userId = jsonObject.optString(NX_TO_USER);
        } else {
            userId = jsonObject.optString(NX_FROM_USER);
        }
    }

    public String toStringEncodedJsonObject() {
        return eventType.toStringEncodedJsonObject(this);
    }

    public NexusEventType getEventType() {
        return eventType;
    }

    public EventData getEventData() {
        return eventData;
    }

    public List<String> getTopics() {
        return topics;
    }

    public String getGuid() {
        return guid;
    }

    public String getUserId() {
        return userId;
    }

    static class Builder {
        final NexusEventType eventName;
        @Nullable EventData eventData;
        @Nullable List<String> topics;
        @Nullable String userId;

        Builder(NexusEventType name) {
            eventName = name;
        }

        public Builder withTopics(List<String> topics) {
            this.topics = topics;
            return this;
        }

        public Builder withEventData(EventData eventData) {
            this.eventData = eventData;
            return this;
        }

        public Builder withUserId(String userId) {
            this.userId = userId;
            return this;
        }

        public NexusEvent build() {
            return new NexusEvent(this);
        }
    }

    public static NexusEvent getAdminIsTypingEvent(String conversationId, String adminId, String adminName,
                                                   String adminAvatarUrl, String userId) {
        EventData eventData = new EventData(4);
        eventData.put(CONVERSATION_ID, conversationId);
        eventData.put(ADMIN_ID, adminId);
        eventData.put(ADMIN_NAME, adminName);
        eventData.put(ADMIN_AVATAR, adminAvatarUrl);
        return new NexusEvent.Builder(AdminIsTyping)
                .withEventData(eventData)
                .withUserId(userId)
                .withTopics(conversationTopics(conversationId))
                .build();
    }

    public static NexusEvent getAdminIsTypingNoteEvent(String conversationId, String adminId, String adminName,
                                                       String adminAvatarUrl, String userId) {
        EventData eventData = new EventData(4);
        eventData.put(ADMIN_ID, adminId);
        eventData.put(CONVERSATION_ID, conversationId);
        eventData.put(ADMIN_NAME, adminName);
        eventData.put(ADMIN_AVATAR, adminAvatarUrl);

        return new NexusEvent.Builder(AdminIsTypingANote)
                .withEventData(eventData)
                .withTopics(conversationTopics(conversationId))
                .withUserId(userId)
                .build();
    }

    public static NexusEvent getUserIsTypingEvent(String conversationId, String userId) {
        EventData eventData = new EventData(1);
        eventData.put(CONVERSATION_ID, conversationId);
        return new NexusEvent.Builder(UserIsTyping)
                .withEventData(eventData)
                .withTopics(conversationTopics(conversationId))
                .withUserId(userId)
                .build();
    }

    public static NexusEvent getConversationSeenEvent(String conversationId, String userId) {
        EventData eventData = new EventData(1);
        eventData.put(CONVERSATION_ID, conversationId);
        return new NexusEvent.Builder(ConversationSeen)
                .withEventData(eventData)
                .withTopics(conversationTopics(conversationId))
                .withUserId(userId)
                .build();
    }

    public static NexusEvent getConversationSeenAdminEvent(String conversationId, String userId) {
        EventData eventData = new EventData(1);
        eventData.put(CONVERSATION_ID, conversationId);
        return new NexusEvent.Builder(UserContentSeenByAdmin)
                .withEventData(eventData)
                .withTopics(conversationTopics(conversationId))
                .withUserId(userId)
                .build();
    }

    public static NexusEvent getNewCommentEvent(String conversationId, String userId) {
        EventData eventData = new EventData(1);
        eventData.put(CONVERSATION_ID, conversationId);
        return new NexusEvent.Builder(NewComment)
                .withEventData(eventData)
                .withTopics(conversationTopics(conversationId))
                .withUserId(userId)
                .build();
    }

    public static NexusEvent getNewNoteEvent(String conversationId, String adminId) {
        EventData eventData = new EventData(2);
        eventData.put(ADMIN_ID, adminId);
        eventData.put(CONVERSATION_ID, conversationId);
        return new NexusEvent.Builder(NewNote)
                .withEventData(eventData)
                .build();
    }

    public static NexusEvent getConversationAssignedEvent(String conversationId, String adminId, String assigneeId) {
        EventData eventData = new EventData(3);
        eventData.put(ADMIN_ID, adminId);
        eventData.put(CONVERSATION_ID, conversationId);
        eventData.put(ASSIGNEE_ID, assigneeId);
        return new NexusEvent.Builder(ConversationAssigned)
                .withEventData(eventData)
                .build();
    }

    public static NexusEvent getConversationClosedEvent(String conversationId, String adminId) {
        EventData eventData = new EventData(2);
        eventData.put(ADMIN_ID, adminId);
        eventData.put(CONVERSATION_ID, conversationId);
        return new NexusEvent.Builder(ConversationClosed)
                .withEventData(eventData)
                .build();
    }

    public static NexusEvent getConversationReopenedEvent(String conversationId, String adminId) {
        EventData eventData = new EventData(2);
        eventData.put(ADMIN_ID, adminId);
        eventData.put(CONVERSATION_ID, conversationId);
        return new NexusEvent.Builder(ConversationReopened)
                .withEventData(eventData)
                .build();
    }

    public static NexusEvent getUserPresenceEvent() {
        return new NexusEvent.Builder(UserPresence)
                .build();
    }

    public static NexusEvent getSubscribeEvent(List<String> topics) {
        return new NexusEvent.Builder(Subscribe)
                .withTopics(topics)
                .build();
    }

    public static NexusEvent getUnsubscribeEvent(List<String> topics) {
        return new NexusEvent.Builder(Unsubscribe)
                .withTopics(topics)
                .build();
    }

    private static List<String> conversationTopics(String conversationId) {
        String subTopic = conversationId;

        if (TextUtils.isEmpty(conversationId)) {
            subTopic = "new";
        }

        return Collections.singletonList(TOPIC_PREFIX_CONVERSATION + subTopic);
    }
}
