/*
 * Copyright 2017 Gupshup.io. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.gupshup.maven;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.Socket;
import java.net.URL;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.MojoExecution;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;
import org.eclipse.jetty.maven.plugin.JettyRunMojo;
import org.eclipse.jetty.server.Server;
import org.json.JSONObject;

import io.gupshup.maven.utils.Constants;
import io.gupshup.maven.utils.Constants.TCPTunnelConsts;
import io.gupshup.maven.utils.HttpUtils;
import io.gupshup.maven.utils.SMAPIWrapper;

/**
 * @author Abhishek Nama
 */
@Mojo(name = "run", threadSafe = true, defaultPhase = LifecyclePhase.TEST_COMPILE, requiresDependencyResolution = ResolutionScope.TEST)
public class Run extends JettyRunMojo {
    @Parameter(defaultValue = "${project}", readonly = true, required = true)
    private MavenProject project;

    @Parameter(defaultValue = "${mojoExecution}", readonly = true)
    private MojoExecution execution;

    @Parameter(defaultValue = "${project.artifacts}", readonly = true)
    protected Set<Artifact> projectArtifacts;

    @Parameter(defaultValue = "${plugin.artifacts}", readonly = true)
    protected List<String> pluginArtifacts;

    @Parameter(defaultValue = "${project.build.outputDirectory}", required = true)
    protected File classesDirectory;

    @Parameter(property = "port", defaultValue = TCPTunnelConsts.DESTINATION_PORT)
    private String port;

    @Parameter(property = "ngrokUrl")
    private String ngrokUrl;

    @Override
    public void execute() throws MojoExecutionException {
	try {
	    List<?> resources = project.getResources();
	    if (resources.size() < 1)
		throw new MojoExecutionException("Project resources not found.");

	    org.apache.maven.model.Resource resource = (org.apache.maven.model.Resource) resources.get(0);
	    File gsCfgPropsFile = new File(resource.getDirectory() + File.separator + Constants.BOT_PROPERTIES);
	    if (!gsCfgPropsFile.exists())
		throw new MojoExecutionException("gsbot-cfg.properties not found in src" + File.separator + "main"
			+ File.separator + "resources");

	    Properties gsCfgProps = new Properties();
	    gsCfgProps.load(new FileInputStream(gsCfgPropsFile));

	    startBot(server, gsCfgProps.getProperty(Constants.GS_APIKEY), gsCfgProps.getProperty(Constants.BOT_NAME));

	    System.setProperty("jetty.port", port);

	    super.scanIntervalSeconds = 1;
	    super.project = this.project;
	    super.execution = this.execution;
	    super.reload = "automatic";
	    super.projectArtifacts = this.projectArtifacts;
	    super.pluginArtifacts = this.pluginArtifacts;
	    super.classesDirectory = this.classesDirectory;

	    super.execute();
	} catch (Exception e) {
	    throw new MojoExecutionException("", e);
	}
    }

    public void startBot(Server server, String apiKey, String botName) {
	TCPTunnel tcpTunnel = new TCPTunnel();

	Thread th = new Thread(new Runnable() {
	    @Override
	    public void run() {
		try {
		    Thread.sleep(3000);
		    if (ngrokUrl != null && !ngrokUrl.trim().equals("")) {
			setCallback(apiKey, botName, ngrokUrl + TCPTunnelConsts.BOT_SERVLET_PATH);
		    } else {
			tcpTunnel.startTunnel(apiKey, botName, Integer.parseInt(port));
		    }
		} catch (InterruptedException e) {
		    if (e.getMessage() != null)
			System.err.println("[ERROR] " + e.getMessage());
		    tcpTunnel.stopTunnel(server);

		    try {
			server.stop();
		    } catch (Exception exp) {

		    }

		    return;
		}
	    }
	});

	th.start();
    }

    public static String userId = null;
    public static String devId = null;

    public static void setCallback(String apiKey, String botName, String callback) {
	if (userId == null) {
	    userId = SMAPIWrapper.getUserId(apiKey);
	    System.out.println("[INFO] ");
	}

	if (devId == null) {
	    devId = SMAPIWrapper.getDevId(apiKey, userId, botName);
	}

	SMAPIWrapper.setCallback(apiKey, userId, devId, botName, callback);

	System.out.println("[INFO] --- Bot Started ---");
	System.out.println("[INFO] You can use your bot on Gupshup Proxy Bot using: proxy " + botName + "$" + devId
		+ "\n\n" + "------------------------------------------------------------------------\n");
    }
}

class TCPTunnel implements Runnable {
    private Thread tunnelThread;
    private AtomicBoolean keepRunning;

    private int tunnelServerPort = -1;
    private int destinationPort = -1;
    private String tunnelUrl = "";
    private Socket tunnelServerSocket;
    private Socket botServerSocket;

    private void getPort(String apiKey, String botName) throws InterruptedException {
	try {
	    System.out.println("[INFO] Attempting to start bot.");
	    tunnelServerPort = -1;
	    URL url = new URL(TCPTunnelConsts.LOCAL_TUNNEL_NEW);
	    HttpURLConnection con = (HttpURLConnection) url.openConnection();
	    con.setRequestMethod("GET");

	    con.setConnectTimeout(30000);
	    con.setReadTimeout(30000);

	    JSONObject resp = new JSONObject(HttpUtils.convertStreamToString(con.getInputStream()));
	    tunnelUrl = resp.getString("url");
	    tunnelServerPort = resp.getInt("port");
	    Run.setCallback(apiKey, botName, tunnelUrl + TCPTunnelConsts.BOT_SERVLET_PATH);
	} catch (Exception e) {
	    e.printStackTrace();
	    throw new InterruptedException("[ERROR] Unable to start bot.");
	}
    }

    public void startTunnel(String apiKey, String botName, int port) throws InterruptedException {
	destinationPort = port;
	keepRunning = new AtomicBoolean(true);

	getPort(apiKey, botName);
	while (keepRunning.get()) {
	    try {
		tunnelServerSocket = new Socket(TCPTunnelConsts.LOCAL_TUNNEL_HOST, tunnelServerPort);
	    } catch (IOException e) {
		getPort(apiKey, botName);
		continue;
	    }

	    tunnelThread = new Thread(this);
	    tunnelThread.start();

	    tunnelThread.join();
	}

	System.out.println("[INFO] Exiting from tunnel.");
    }

    public void stopTunnel(Server server) {
	if (keepRunning != null)
	    keepRunning.set(false);
	try {
	    server.stop();
	} catch (Exception e) {
	    System.err.println("[ERROR] Exception in stopping bot server.");
	    e.printStackTrace();
	}
    }

    public void run() {
	InputStream tunnelServerIn;
	OutputStream tunnelServerOut;
	InputStream botServerIn;
	OutputStream botServerOut;

	try {
	    botServerSocket = new Socket(TCPTunnelConsts.DESTINATION_HOST, destinationPort);
	    botServerSocket.setKeepAlive(true);
	    tunnelServerSocket.setKeepAlive(true);

	    tunnelServerIn = tunnelServerSocket.getInputStream();
	    tunnelServerOut = tunnelServerSocket.getOutputStream();
	    botServerIn = botServerSocket.getInputStream();
	    botServerOut = botServerSocket.getOutputStream();
	} catch (IOException ioe) {
	    System.err.println("[ERROR] Bot Server is not started.");
	    connectionBroken();
	    return;
	}

	ForwardThread tunnelServerForward = new ForwardThread(tunnelServerIn, botServerOut);
	tunnelServerForward.start();
	ForwardThread botServerForward = new ForwardThread(botServerIn, tunnelServerOut);
	botServerForward.start();

	try {
	    tunnelServerForward.join();
	} catch (InterruptedException e) {

	}

	try {
	    botServerForward.join();
	} catch (InterruptedException e) {

	}
    }

    public synchronized void connectionBroken() {
	try {
	    tunnelServerSocket.close();
	} catch (Exception e) {

	}

	try {
	    botServerSocket.close();
	} catch (Exception e) {

	}
    }

    class ForwardThread extends Thread {
	private static final int BUFFER_SIZE = 8192;

	InputStream mInputStream;
	OutputStream mOutputStream;

	public ForwardThread(InputStream aInputStream, OutputStream aOutputStream) {
	    mInputStream = aInputStream;
	    mOutputStream = aOutputStream;
	}

	public void run() {
	    byte[] buffer = new byte[BUFFER_SIZE];

	    try {
		while (true) {
		    int bytesRead = mInputStream.read(buffer);
		    if (bytesRead == -1)
			break;
		    mOutputStream.write(buffer, 0, bytesRead);
		    mOutputStream.flush();
		}
	    } catch (IOException e) {

	    }

	    connectionBroken();
	}
    }
}