/*
 * Copyright 2017 Gupshup.io. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.gupshup.maven;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.List;
import java.util.Properties;

import org.apache.maven.model.Resource;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;
import org.eclipse.jgit.api.CheckoutCommand;
import org.eclipse.jgit.api.CreateBranchCommand.SetupUpstreamMode;
import org.eclipse.jgit.api.FetchCommand;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.InitCommand;
import org.eclipse.jgit.api.RemoteAddCommand;
import org.eclipse.jgit.transport.URIish;
import org.json.JSONObject;

import io.gupshup.maven.utils.Constants;
import io.gupshup.maven.utils.Constants.SMAPIConsts;
import io.gupshup.maven.utils.SMAPIWrapper;

/**
 * @author Abhishek Nama
 */
@Mojo(name = "create", threadSafe = true, defaultPhase = LifecyclePhase.INITIALIZE, requiresDependencyResolution = ResolutionScope.COMPILE_PLUS_RUNTIME)
public class Create extends AbstractMojo {
    @Parameter(defaultValue = "${project}", readonly = true, required = true)
    private MavenProject project;

    @Parameter(property = "apiKey", required = true)
    private String apiKey;

    @Parameter(property = "botName", required = true)
    private String botName;

    @Parameter(property = "description", required = false)
    private String description;

    @Override
    public void execute() throws MojoExecutionException {
	try {
	    List<?> resources = project.getResources();
	    if (resources.size() < 1)
		throw new MojoExecutionException("Project resources not found.");

	    Resource resource = (Resource) resources.get(0);
	    File gsCfgPropsFile = new File(resource.getDirectory() + File.separator + Constants.BOT_PROPERTIES);
	    if (!gsCfgPropsFile.exists())
		throw new MojoExecutionException("gsbot-cfg.properties not found in src" + File.separator + "main"
			+ File.separator + "resources");

	    Properties gsCfgProps = new Properties();
	    gsCfgProps.load(new FileInputStream(gsCfgPropsFile));

	    getLog().info("");
	    getLog().info("------------------------------------------------------------------------ ");
	    getLog().info("Creating bot " + botName + " for " + project.getName());
	    getLog().info("------------------------------------------------------------------------ ");
	    getLog().info("");

	    getLog().info("Configuring gsbot-cfg.properties");

	    JSONObject createBotResp = SMAPIWrapper.createBot(apiKey, botName, description);
	    if (createBotResp.getInt("status") != 200)
		throw new MojoExecutionException(createBotResp.getString("body"));

	    String botKey = new JSONObject(createBotResp.getString("body")).getString("botkey");

	    gsCfgProps.setProperty("gupshup.dev.apikey", apiKey);
	    gsCfgProps.setProperty("gupshup.bot.name", botName);
	    gsCfgProps.setProperty("gupshup.bot.key", botKey);
	    gsCfgProps.store(new FileOutputStream(gsCfgPropsFile), null);

	    getLog().info("Creating bot repository.");

	    SMAPIWrapper.createGit(botKey);

	    getLog().info("Synching with bot repository.");

	    String userId = SMAPIWrapper.getUserId(apiKey);
	    if (userId != null) {
		String gitUrl = SMAPIConsts.GIT_URL.replace("__USERID", userId).replace("__APIKEY", apiKey)
			.replace("__BOTKEY", botKey);
		boolean isSuccess = cloneBotRepo(gitUrl);
		if (!isSuccess) {
		    getLog().info("");
		    getLog().warn("Failed to sync with bot repository.");
		}
	    } else {
		getLog().info("");
		getLog().error("Failed to sync with bot repository.");
	    }

	    getLog().info("");
	    getLog().info("Bot created successfully!");
	    getLog().info("");
	} catch (Exception e) {
	    throw new MojoExecutionException("", e);
	}
    }

    private boolean cloneBotRepo(String gitUrl) {
	try {
	    InitCommand initCommand = Git.init().setDirectory(project.getBasedir());
	    Git git = initCommand.call();

	    RemoteAddCommand remoteAddCommand = git.remoteAdd();
	    remoteAddCommand.setName("origin");
	    remoteAddCommand.setUri(new URIish(gitUrl));
	    remoteAddCommand.call();

	    FetchCommand fetchCommand = git.fetch();
	    fetchCommand.call();

	    CheckoutCommand checkoutCommand = git.checkout();
	    checkoutCommand.setCreateBranch(true).setName("master").setUpstreamMode(SetupUpstreamMode.TRACK)
		    .setStartPoint("origin/master").call();

	    return true;
	} catch (Exception e) {
	    e.printStackTrace();
	    return false;
	}
    }
}