/*
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "[]"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2019 Rujal Shrestha
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 *
 */
package io.gitlab.rujal_sh.commons;

import com.zaxxer.hikari.HikariDataSource;
import io.gitlab.rujal_sh.annotation.StrategyType;
import io.gitlab.rujal_sh.annotation.internal.TenantHolder;
import io.gitlab.rujal_sh.annotation.internal.DataSourceConfiguration;
import io.gitlab.rujal_sh.commons.utils.Constants;
import io.gitlab.rujal_sh.database.domain.DataSourceConfig;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.hibernate.cfg.Environment;
import org.hibernate.context.spi.CurrentTenantIdentifierResolver;
import org.hibernate.engine.jdbc.connections.spi.AbstractDataSourceBasedMultiTenantConnectionProviderImpl;
import org.hibernate.engine.jdbc.connections.spi.MultiTenantConnectionProvider;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.boot.jdbc.DataSourceBuilder;
import org.springframework.data.repository.CrudRepository;
import org.springframework.jdbc.datasource.ConnectionHandle;
import org.springframework.jdbc.datasource.ConnectionHolder;
import org.springframework.orm.jpa.DefaultJpaDialect;
import org.springframework.orm.jpa.LocalContainerEntityManagerFactoryBean;
import org.springframework.orm.jpa.vendor.HibernateJpaVendorAdapter;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.TransactionSynchronizationManager;
import org.springframework.web.context.support.GenericWebApplicationContext;

import javax.sql.DataSource;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author Rujal Shrestha
 */
@RequiredArgsConstructor
@Aspect
@SuppressWarnings("unchecked")
public class DBTenantHelper extends SqlExecutions {

    private final DataSource dataSource;
    private final BeanFactory beanFactory;
    private final GenericWebApplicationContext context;
    private final JpaHibernateProperties jpaHibernateProperties;
    private final CurrentTenantIdentifierResolver currentTenantIdentifierResolver;
    private final Optional<TenantDataSource> optionalTenantDataSource;
    private final MultiTenantConnectionProvider multiTenantConnectionProvider;

    private PersistenceJPAConfig persistenceJPAConfig;
    private DataSourceConfig dataSourceConfig;

    @Around("execution(* org.springframework.data.repository.CrudRepository+.save(..)) && args(entity)")
    public Object test(ProceedingJoinPoint proceedingJoinPoint, Object entity) throws Throwable {
        if (currentTenantIdentifierResolver.resolveCurrentTenantIdentifier().equalsIgnoreCase("public")) {
            String userName = null;

            List<Annotation> annotationList = Arrays.asList(entity.getClass().getAnnotations()).stream().filter(annotation -> annotation instanceof TenantHolder).collect(Collectors.toList());
            if (!annotationList.isEmpty()) {
                persistenceJPAConfig = new PersistenceJPAConfig(jpaHibernateProperties, dataSource, multiTenantConnectionProvider, currentTenantIdentifierResolver, context);

                try {
                    Method methodUserName = entity.getClass().getDeclaredMethod("getName");
                    userName = String.valueOf(methodUserName.invoke(entity));
                } catch (NoSuchMethodException ne) {
                    Method methodUserName = entity.getClass().getDeclaredMethod("getUsername");
                    userName = String.valueOf(methodUserName.invoke(entity));
                }

                CrudRepository crudRepository = (CrudRepository) proceedingJoinPoint.getThis();

                List<Object> result = new ArrayList<>();
                crudRepository.findAll().forEach(result::add);

                switch (Constants.strategy) {
                    case DATABASE:
                        DataSourceConfig dbConfig = getDataSourceConfig(entity);
                        result = result.stream().filter(entityFromRepo -> checkDbConfiguration(entityFromRepo, dbConfig)).collect(Collectors.toList());

                        if (result.isEmpty()) {
                            performDatabaseAction(entity, dbConfig, userName);
                        }
                        proceedingJoinPoint.proceed();
                        break;

                    case SCHEMA:
                        createTenantSchema(userName);
                        proceedingJoinPoint.proceed();
                        break;
                }
                createTables(userName, getValue(entity, "dataSourceConfig", DataSourceConfig.class));
            }
        }
        return proceedingJoinPoint.proceed();
    }

    private void performDatabaseAction(Object entity, DataSourceConfig dbConfig, String userName) throws SQLException {
        if (!dbConfig.getGenerate()) {
            testDbConnection(dbConfig);
        } else {
            dataSourceConfig = dbConfig = getTenantWiseDataSourceConfig(userName);
            createDatabase(userName);
            optionalTenantDataSource.get().dataSources.put(dbConfig.getName(), createDataSource(dataSourceConfig));
            setValue(entity, "dataSourceConfig", dbConfig, DataSourceConfig.class);
        }
    }

    private void createTenantSchema(String tenantName) throws SQLException {
        Connection connection = dataSource.getConnection();
        connection.createStatement()
                .execute(String.format("CREATE SCHEMA IF NOT EXISTS %s", tenantName));
        connection.setSchema(tenantName);
    }

    private DataSourceConfig getTenantWiseDataSourceConfig(String tenantId) {
        HikariDataSource dataSourceH = ((HikariDataSource) dataSource);
        DataSourceConfig dataSourceConfig = DataSourceConfig.builder()
                .driverClassName(dataSourceH.getDriverClassName())
                .name(tenantId)
                .username(dataSourceH.getUsername())
                .generate(true)
                .password(dataSourceH.getPassword())
                .initialize(true)
                .url(generateUrl(tenantId, dataSourceH.getJdbcUrl()))
                .build();
        return dataSourceConfig;
    }

    private String generateUrl(String tenantId, String jdbcUrl) {
        List<String> strings = new ArrayList<>(Arrays.asList(jdbcUrl.split("/")));
        strings.add(strings.size(), tenantId);
        strings.remove(strings.size() - 2);
        return String.join("/", strings);
    }

    private void testDbConnection(DataSourceConfig dbConfig) {
        try {
            Class.forName(dbConfig.getDriverClassName());
            Connection connection = DriverManager.getConnection(dbConfig.getUrl(), dbConfig.getUsername(), dbConfig.getPassword());
            if (connection == null) {
                throw new RuntimeException(String.format("Driver connection to URL %s failed", dbConfig.getUrl()));
            }
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        } catch (SQLException throwables) {
            throwables.printStackTrace();
        }
    }


    private boolean checkDbConfiguration(Object entityRepo, DataSourceConfig dbConfig) {
        DataSourceConfig dataSourceConfig = getDataSourceConfig(dbConfig);
        if (dataSourceConfig != null) {
            DataSourceConfig dataSource = (DataSourceConfig) entityRepo;
            if (dataSource.getUsername().equalsIgnoreCase(dbConfig.getUsername())) {
                return true;
            }
        }
        return false;
    }

    private DataSourceConfig getDataSourceConfig(Object entity) {
        try {
            String fieldName = getFieldNameWithAnnotationDataSourceConfiguration(entity);
            Method methodId = entity.getClass().getSuperclass().getDeclaredMethod("get" + fieldName);
            Object dbConfig = methodId.invoke(entity);
            if (dbConfig != null) {
                DataSourceConfig dataSourceConfig = (DataSourceConfig) dbConfig;
                if (dataSourceConfig.getUrl() == null || dataSourceConfig.getUrl().trim().equalsIgnoreCase("")) {
                    dataSourceConfig.setGenerate(true);
                }
            } else {
                dataSourceConfig = DataSourceConfig.builder().generate(true).build();
            }
        } catch (Exception e) {
            throw new RuntimeException(e.getMessage());
        }
        return dataSourceConfig;
    }

    private void createDatabase(String tenantName) throws SQLException {
        DataSourceBuilder dataSourceBuilder = DataSourceBuilder
                .create().driverClassName(((HikariDataSource) dataSource).getDriverClassName())
                .username(((HikariDataSource) dataSource).getUsername())
                .password(((HikariDataSource) dataSource).getPassword())
                .url(((HikariDataSource) dataSource).getJdbcUrl());
        DataSource ds = dataSourceBuilder.build();
        Connection connection = ds.getConnection();
        Statement statement = connection.createStatement();
        if (!tenantName.equalsIgnoreCase("public")) {
            try {
                statement.execute(String.format("create database %s", tenantName));
            } catch (Exception e) {
                //already exists
            }
        }
    }

    private String getFieldNameWithAnnotationDataSourceConfiguration(Object entity) {
        List<Field> fields = Arrays.asList(entity.getClass().getSuperclass().getDeclaredFields());
        if (fields.size() != 0) {
            int count = 0;
            Field field = null;
            for (Field f : fields) {
                Optional<Annotation> schemaFields = Arrays.asList(f.getAnnotations()).stream().filter(annotation -> annotation instanceof DataSourceConfiguration).findFirst();
                if (schemaFields.isPresent()) {
                    field = f;
                    count++;
                }
            }
            if (count != 1) {
                throw new RuntimeException("Fields cannot be annotated with '@DataSourceConfiguration' more than once");
            } else {
                String fieldName = new StringBuilder().append(field.getName().substring(0, 1).toUpperCase()).append(field.getName().substring(1)).toString();
                return fieldName;
            }
        } else {
            throw new RuntimeException("Consider extending class 'DataSourceComponent' in class annotated with @TenantHolder");
        }
    }

    private String createBean(String tenantId, DataSourceConfig dbConfig) {
        String beanName = "entityManagerFactory" + tenantId;
        context.registerBean(beanName, CustomEntityManagerFactoryBean.class, () -> customEntityManagerFactoryBean(tenantId, dbConfig));
        return beanName;
    }

    private void createTables(String userName, DataSourceConfig dbConfig) {
        if (!userName.equalsIgnoreCase("public")) {
            String beanName = createBean(userName, dbConfig);
            CustomEntityManagerFactoryBean l = beanFactory.getBean(CustomEntityManagerFactoryBean.class);
            context.removeBeanDefinition(beanName);
        }
    }

    CustomEntityManagerFactoryBean customEntityManagerFactoryBean(String tenantId, DataSourceConfig dbConfig) {

        DataSource dataSource_tenant = getTenantWiseDataSource(dbConfig);
        MultiTenantConnectionProvider multiTenantConnectionProvider = new AbstractDataSourceBasedMultiTenantConnectionProviderImpl() {
            @Override
            protected DataSource selectAnyDataSource() {
                return selectDataSource(tenantId);
            }

            @Override
            protected DataSource selectDataSource(String tenantIdentifier) {
                return dataSource_tenant;
            }
        };
        LocalContainerEntityManagerFactoryBean localContainerEntityManagerFactoryBean = persistenceJPAConfig.entityManagerFactory();

        CustomEntityManagerFactoryBean customEntityManagerFactoryBean = new CustomEntityManagerFactoryBean();
        Map<String, Object> jpaPropertyMap = localContainerEntityManagerFactoryBean.getJpaPropertyMap();

        if (Constants.strategy.equals(StrategyType.SCHEMA)) {
            jpaPropertyMap.put("hibernate.default_schema", tenantId);
            customEntityManagerFactoryBean.setDataSource(dataSource);
        } else {
            if (!dbConfig.getGenerate()) {
                jpaPropertyMap.put(Environment.USER, dbConfig.getUsername());
                jpaPropertyMap.put(Environment.PASS, dbConfig.getPassword());
                jpaPropertyMap.put(Environment.URL, dbConfig.getUrl());
                jpaPropertyMap.put(Environment.DRIVER, dbConfig.getDriverClassName());
            } else {
                try {
                    jpaPropertyMap.put(Environment.URL, dbConfig.getUrl());
                } catch (ClassCastException classCastException) {
                    //do nothing
                }
            }
            jpaPropertyMap.put(Environment.MULTI_TENANT_CONNECTION_PROVIDER, multiTenantConnectionProvider);
            customEntityManagerFactoryBean.setDataSource(dataSource_tenant);
        }
        customEntityManagerFactoryBean.setJpaPropertyMap(jpaPropertyMap);

        customEntityManagerFactoryBean.setJpaVendorAdapter(new HibernateJpaVendorAdapter());
        customEntityManagerFactoryBean.setPackagesToScan(Constants.basePackages);
        return customEntityManagerFactoryBean;
    }

    private DataSource getTenantWiseDataSource(DataSourceConfig dbConfig) {
        if (dbConfig != null) {
            DataSourceBuilder factory = DataSourceBuilder
                    .create().driverClassName(dbConfig.getDriverClassName())
                    .username(dbConfig.getUsername())
                    .password(dbConfig.getPassword())
                    .url(dbConfig.getUrl());
            return factory.build();
        }else {
            return null;
        }
    }
    // pass tenenta id and get datasource

    private <T> T getValue(Object entity, String fieldName, Class<T> classType) {
        try {
            fieldName = fieldName.substring(0, 1).toUpperCase() + fieldName.substring(1);
            Method methodId = entity.getClass().getSuperclass().getDeclaredMethod("get" + fieldName);
            T value = (T) methodId.invoke(entity);
            return value;
        } catch (Exception e) {
            return null;
        }
    }

    private <T> void setValue(Object entity, String fieldName, Object value, Class<T> classType) {
        try {
            Field field = entity.getClass().getSuperclass().getDeclaredField(fieldName);
            field.setAccessible(true);
            field.set(entity, value);
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e.getMessage());
        }
    }
}
