package com.tunetalk.ttdealersdk.ocr;

import android.app.Activity;
import android.content.DialogInterface;
import android.os.Handler;

import androidx.annotation.NonNull;
import androidx.appcompat.app.AlertDialog;

import com.microblink.entities.recognizers.Recognizer;
import com.microblink.entities.recognizers.RecognizerBundle;
import com.microblink.entities.recognizers.blinkbarcode.barcode.BarcodeRecognizer;
import com.microblink.entities.recognizers.blinkid.generic.AnonymizationMode;
import com.microblink.entities.recognizers.blinkid.generic.BlinkIdRecognizer;
import com.microblink.entities.recognizers.blinkid.generic.classinfo.Type;
import com.microblink.entities.recognizers.blinkid.mrtd.MrzResult;
import com.microblink.image.Image;
import com.microblink.uisettings.ActivityRunner;
import com.microblink.uisettings.UISettings;
import com.tunetalk.ttdealersdk.ocr.entity.ScanResultEntity;
import com.tunetalk.ttdealersdk.ocr.entity.ScanResultItemEntity;
import com.tunetalk.ttdealersdk.util.Common;
import com.tunetalk.ttdealersdk.util.FileUtils;
import com.tunetalk.ttdealersdk.util.Make;

import java.util.Locale;

import static com.microblink.MicroblinkSDK.getApplicationContext;

public class OCRHelper
{
    private static final String DOCUMENT_FILE_PATH = FileUtils.ORC_DIR + "/identity.jpg";
    private static OCRHelper mInstance;

    private Activity mActivity;

    public static synchronized OCRHelper get ()
    {
        if (mInstance == null)
            mInstance = new OCRHelper();

        return mInstance;
    }

    /**
     * Configure MyKad / iKad / MyTentera / MYPR recognizer
     *
     * @return RecognizerBundle
     */
    public RecognizerBundle buildBlinkIdRecognizerElement()
    {

        BlinkIdRecognizer element = new BlinkIdRecognizer();
        OCRSettings.configureImageSettings(element);
        return prepareRecognizerBundle(element);

    }

    /**
     * Configure MRTD recognizer (e.g.: For passport, imm13)
     *
     * @return RecognizerBundle
     */
    public RecognizerBundle buildMrtdElement ()
    {
        BlinkIdRecognizer mrtdRecognizer = new BlinkIdRecognizer();
        mrtdRecognizer.setAnonymizationMode(AnonymizationMode.None);
        mrtdRecognizer.setReturnFullDocumentImage(true);
        OCRSettings.configureImageSettings(mrtdRecognizer);

        return prepareRecognizerBundle(mrtdRecognizer);
    }

    /**
     * Configure barcode recognizer
     *
     * @return RecognizerBundle
     */
    public RecognizerBundle buildBarCodeElement ()
    {
        BarcodeRecognizer barcode = new BarcodeRecognizer();
        barcode.setScanCode39(true);
        barcode.setScanCode128(true);
        barcode.setScanInverse(true);
        barcode.setScanAztecCode(true);
        barcode.setScanDataMatrix(true);
        barcode.setScanEan13(true);
        barcode.setScanEan8(true);
        barcode.setScanItf(true);
        barcode.setScanPdf417(true);
        barcode.setScanQrCode(true);
        barcode.setScanUpca(true);
        barcode.setScanUpce(true);

        return prepareRecognizerBundle(barcode);
    }

    /**
     * Add all the required recognizers and form a bundle
     *
     * @param recognizers
     * @return RecognizerBundle
     */
    private RecognizerBundle prepareRecognizerBundle (@NonNull Recognizer<?>... recognizers)
    {
        return new RecognizerBundle(recognizers);
    }

    /**
     * Start the scan activity
     *
     * @param activity
     * @param settings
     * @param resultCode
     */
    public void scan (Activity activity, @NonNull UISettings settings, int resultCode)
    {
        mActivity = activity;
        OCRSettings.configureScanSettings(settings, null);
        ActivityRunner.startActivityForResult(activity, resultCode, settings);
    }

    /**
     * Retrieve the scanning data and images
     *
     * @param mRecognizerBundle
     * @param result
     */
    public void getResult (RecognizerBundle mRecognizerBundle, final OnScanResult result)
    {
        final ScanResultEntity mResultEntity = new ScanResultEntity();
        final ScanResultItemEntity itemEntity = new ScanResultItemEntity();

        if (mRecognizerBundle == null)
        {
            return;
        }

        for (Recognizer r : mRecognizerBundle.getRecognizers()) {
            if (r instanceof BlinkIdRecognizer) {
                BlinkIdRecognizer.Result scanResult = (BlinkIdRecognizer.Result) r.getResult();

                if (scanResult.getResultState() == Recognizer.Result.State.Valid) {

                        if (scanResult.getClassInfo().getType().equals(Type.MyKad) || scanResult.getClassInfo().getType().equals(Type.MyPR)) {

                            boolean isMyKad = scanResult.getClassInfo().getType().equals(Type.MyKad);

                            try {
                                String NRIC = scanResult.getDocumentNumber().replaceAll("-", "");

                                if (Common.isValidNRIC(NRIC)) {
                                    itemEntity.setDocumentType(isMyKad ? DocumentType.MYKAD : DocumentType.MYPR);
                                    itemEntity.setFullName(Common.ProcessName(scanResult.getFullName().replace("\r\n", " ").replace("\n", " ")));
                                    itemEntity.setNric(NRIC);
                                    itemEntity.setGender(Common.getGenderFromNRIC(NRIC));
                                    itemEntity.setDob(Common.getDOBFromNRIC(NRIC));

                                    populateAddress(scanResult.getAddress(), itemEntity);

                                } else {
                                    showErrorDialog();
                                }

                            } catch (Exception e) {
                                e.printStackTrace();
                            }

                            compressScannedImage(scanResult.getFullDocumentImage());

                            if (isMyKad) {
                                mResultEntity.setMyKad(itemEntity);
                            } else {
                                mResultEntity.setMypr(itemEntity);
                            }


                        } else if (scanResult.getClassInfo().getType().equals(Type.iKAD)) {

                            if (Common.isValidAlphaNumeric(scanResult.getDocumentNumber().replaceAll("<", ""))) {
                                String rawDob = scanResult.getDateOfBirth().getOriginalDateString();
                                String mDOB = Common.reformatDateForScanEntity("dd/MM/yyyy", rawDob);

                                itemEntity.setDocumentType(DocumentType.IKAD)
                                        .setFullName(scanResult.getFullName())
                                        .setPassportNumber(scanResult.getDocumentNumber().replaceAll("<", ""))
                                        .setNationality(scanResult.getNationality())
                                        .setGender(Common.getGenderFromScanResult(scanResult.getSex()))
                                        .setDob(mDOB);

                                populateAddress(scanResult.getAddress(), itemEntity);

                                for (String countryCode : Locale.getISOCountries()) {
                                    Locale obj = new Locale("", countryCode);

                                    if (obj.getDisplayCountry().equalsIgnoreCase(scanResult.getNationality()))
                                        itemEntity.setNationality(obj.getDisplayName());
                                }

                            } else {
                                showErrorDialog();
                            }

                            compressScannedImage(scanResult.getFullDocumentImage());
                            mResultEntity.setiKad(itemEntity);

                        } else if (scanResult.getClassInfo().getType().equals(Type.MyTentera)) {

                            itemEntity.setDocumentType(DocumentType.ARMY_ID);

                            if (scanResult.getDocumentNumber() != null && !scanResult.getDocumentNumber().isEmpty()) {
                                String NRIC = scanResult.getDocumentNumber().replaceAll("-", "");
                                String MYTENTERA = scanResult.getPersonalIdNumber().replaceAll("-", "");

                                if (Common.isValidMyTentera(MYTENTERA)) {
                                    itemEntity.setFullName(Common.ProcessName(scanResult.getFullName().replace("\r\n", " ").replace("\n", " ")));
                                    itemEntity.setMyTentera(MYTENTERA);
                                    itemEntity.setNric(NRIC);
                                    itemEntity.setGender(Common.getGenderFromNRIC(NRIC));
                                    itemEntity.setDob(Common.getDOBFromNRIC(NRIC));

                                    populateAddress(scanResult.getAddress(), itemEntity);

                                } else {
                                    showErrorDialog();
                                }
                            }

                            compressScannedImage(scanResult.getFullDocumentImage());
                            mResultEntity.setMyTentera(itemEntity);

                        } else if (scanResult.getMrzResult().isMrzParsed()) {
                            BlinkIdRecognizer.Result mrtd = (BlinkIdRecognizer.Result) r.getResult();
                            MrzResult passport = mrtd.getMrzResult();

                            try {
                                String passportNumber = passport.getDocumentNumber().replaceAll("<", "");

                                if (Common.isValidAlphaNumeric(passportNumber)) {
                                    String rawDob = passport.getDateOfBirth().getOriginalDateString();
                                    String mDOB = Common.reformatDateForScanEntity("yyMMdd", rawDob);

                                    itemEntity
                                            .setFullName(Common.replaceSpecialChar(passport.getSecondaryId()) + " " + Common.replaceSpecialChar(passport.getPrimaryId()))
                                            .setDob(mDOB)
                                            .setGender(Common.getGenderFromScanResult(passport.getGender()))
                                            .setPassportNumber(passportNumber)
                                            .setCountryCode(passport.getNationality());

                                    for (String countryCode : Locale.getISOCountries()) {
                                        Locale obj = new Locale("", countryCode);

                                        if (obj.getISO3Country().equals(passport.getNationality())) {
                                            String mNationality = obj.getDisplayName();

                                            if (passport.getNationality().equals("CHN"))
                                                itemEntity.setNationality(passport.getNationality());
                                            else
                                                itemEntity.setNationality(Common.replaceSpecialChar(mNationality));
                                        }
                                    }
                                } else {
                                    showErrorDialog();
                                }

                                compressScannedImage(mrtd.getFullDocumentImage());

                                switch (passport.getDocumentType()) {
                                    case MRTD_TYPE_MYS_PASS_IMM13P:
                                    case MRTD_TYPE_VISA:
                                        itemEntity.setDocumentType(DocumentType.IMM13);
                                        mResultEntity.setImm13(itemEntity);
                                        break;

                                    case MRTD_TYPE_PASSPORT:
                                    default:
                                        itemEntity.setDocumentType(DocumentType.PASSPORT);
                                        mResultEntity.setMrtd(itemEntity);
                                        break;
                                }
                            } catch (Exception ex) {
                                mResultEntity.setMrtd(itemEntity);
                                ex.printStackTrace();
                            }
                        }
                }

            }
            else if (r instanceof BarcodeRecognizer)
            {
                BarcodeRecognizer.Result bar = (BarcodeRecognizer.Result) r.getResult();
                if (bar.getResultState() == Recognizer.Result.State.Valid)
                {
                    try
                    {
                        itemEntity
                                .setDocumentType(DocumentType.BARCODE)
                                .setScannedBarCode(bar.getStringData());
                    }
                    catch (Exception ex)
                    {
                        ex.printStackTrace();
                    }

                    mResultEntity.setBarcode(itemEntity);
                }
            }
        }

        if (result != null)
        {
            Make.ProgressDialog.Show(mActivity);
            new Handler().postDelayed(new Runnable()
            {
                @Override public void run ()
                {
                    Make.ProgressDialog.Dismiss();

                    //Return image result other than barcode document type
                    if (itemEntity.getDocumentType() != null
                            && ! itemEntity.getDocumentType().equals(DocumentType.BARCODE))
                    {
                        result.onImageResult(FileUtils.getBytesFromFile(DOCUMENT_FILE_PATH));
                    }
                    result.onScanResult(mResultEntity);

                }
            }, 1000);

        }
    }

    private void showErrorDialog() {
        try {
            new AlertDialog.Builder(getApplicationContext())
                    .setTitle("Opps...")
                    .setMessage("Unable to read card data, please try again.")
                    .setCancelable(false)
                    .setPositiveButton("OK", new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
//                            finish();
                        }
                    })
                    .show();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void populateAddress(String addressString, ScanResultItemEntity itemEntity) {
        String[] tokens = addressString.split("\\s");

        for (String s : tokens) {
            if (s.length() == 5) {
                try {
                    s = s.replaceAll("O", "0");
                    s = s.replaceAll("o", "0");
                    s = s.replaceAll("l", "1");
                    s = s.replaceAll("B", "8");

                    Integer.parseInt(s);
                    String address = addressString.substring(0, addressString.indexOf(s)).replaceAll("\n", " ");

                    itemEntity
                            .setPoscode(s)
                            .setAddress(address);

                    break;

                } catch (Exception exp) {
                    exp.printStackTrace();
                }
            }
        }
    }

    public void compressScannedImage(Image image) {
        try {
            FileUtils.Compress(DOCUMENT_FILE_PATH, image.convertToBitmap());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
