package com.litehost.core;

import org.atteo.classindex.ClassIndex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.litehost.annotate.ResourceManager;
import com.litehost.annotate.ResourcePath;

import java.io.IOException;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.net.Socket;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

class Wrapper extends Thread {
    private BlockingQueue<Socket> connections;
    private BlockingQueue<ClientRequest> tasks;
    Map<String, Method> requestMappings = null;
    private int maxThreads;
    private Logger logger;

    Wrapper(BlockingQueue<Socket> connections, BlockingQueue<ClientRequest> tasks, int maxExecutingThreads) {
        this.connections = connections;
        this.tasks = tasks;
        this.maxThreads = maxExecutingThreads;
        this.logger = LoggerFactory.getLogger(this.getClass());
    }

    private void initializeMappings() {
        StringBuilder result = new StringBuilder();
        Response responseCheck = Response.builder().build();
        IncomingRequest requestCheck = new IncomingRequest();
        Iterable<Class<?>> classes = ClassIndex.getAnnotated(ResourceManager.class);
        classes.forEach(clazz -> {
            try {
                clazz.getConstructor();
            } catch (NoSuchMethodException e) {
                throw new NoDefaultConstructorException("Manager class must have default constructor!");
            }
        });
        Map<String, Method> mapping = new ConcurrentHashMap<>(10, 0.75f, this.maxThreads);
        for (Class<?> clazz : classes) {
            Method[] methods = clazz.getMethods();
            for (Method method : methods) {
                Annotation annotation = method.getAnnotation(ResourcePath.class);
                if (annotation != null) {
                    Parameter[] parameters = method.getParameters();
                    if (parameters.length > 0) {
                        if (parameters.length != 1)
                            throw new IllegalArgumentException("Only one Request parameter, or no parameters can be passed!");
                        if (!parameters[0].getType().isInstance(requestCheck))
                            throw new IllegalArgumentException("Parameter must be a Request!");
                    }
                    String resourcePath = (( ResourcePath ) annotation).path();
                    String httpMethod = (( ResourcePath ) annotation).method();
                    result.append(httpMethod);
                    result.append(resourcePath);
                    if (!method.getReturnType().isInstance(responseCheck))
                        throw new NotResponseException("Handling method return type must be a Response!");
                    String signature = result.toString();
                    if (signature.contains("{"))
                        signature = signature.replaceAll("\\{\\w+}", "\\\\w+");
                    mapping.put(signature, method);
                    result = new StringBuilder();
                }
            }
        }
        this.requestMappings = mapping;
        ClientRequest.requestMappings = this.requestMappings;
        System.gc();
    }

    private ClientRequest wrap(Socket connection) {
        ClientRequest clientRequest = null;
        try {
            clientRequest = new ClientRequest(connection);
            clientRequest.readRequest();
        } catch (IOException e) {
            this.logger.debug(e.getMessage());
        }
        return clientRequest;
    }

    @Override
    public synchronized void start() {
        this.setUncaughtExceptionHandler(ExceptionHandlers.stopHandler());
        this.setDaemon(true);
        try {
            this.initializeMappings();
        } catch (Exception e) {
            System.err.println(e.getMessage());
            System.err.println(" === Startup aborted ===\n");
            System.exit(420);
        }
        super.start();
        System.out.println(" === Resource managers mapped ===\n");
    }

    private void runShutdownProcedure() {
        try {
            while (!this.connections.isEmpty())
                this.tasks.put(this.wrap(this.connections.poll()));
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        synchronized (Server.shutdownLock) {
            Server.shutdownLock.notify();
        }
    }

    @Override
    public void run() {
        while (true) {
            try {
                if (interrupted()) {
                    this.runShutdownProcedure();
                    break;
                }
                Socket connection = connections.take();
                tasks.put(this.wrap(connection));
            } catch (InterruptedException e) {
                this.runShutdownProcedure();
                break;
            }
        }
    }
}
