package com.litehost.core;

import java.util.HashMap;
import java.util.Map;

public class Response {
    private String version = "HTTP/1.1";
    private HttpStatus status;
    private String stringStatus;
    private Map<String, String> headers;
    private String body;

    private Response() {
        this.status = null;
        this.stringStatus = null;
        this.headers = null;
        this.body = null;
    }

    Response(String message) {
        this.status = null;
        this.version = Requests.getVersion(message);
        this.stringStatus = Requests.getStatus(message);
        this.headers = Requests.getHeaders(message);
        this.body = Requests.getBody(message);
    }

    public static Builder builder() {
        return new Builder();
    }

    public String getBody() {
        return this.body;
    }

    public HttpStatus getStatus() {
        return this.status;
    }

    public Map<String, String> getHeaders() {
        return this.headers;
    }

    public String getVersion() {
        return this.version;
    }

    public static class Builder {
        private Response rawResponse;

        Builder() {
            this.rawResponse = new Response();
        }

        public Builder version(String version) {
            this.rawResponse.version = version;
            return this;
        }

        public Builder status(HttpStatus status) {
            this.rawResponse.status = status;
            return this;
        }

        public Builder headers(Map<String, String> headers) {
            this.rawResponse.headers = headers;
            this.rawResponse.headers.put("Connection", "Closed");
            return this;
        }

        public Builder header(String name, String value) {
            if (this.rawResponse.headers == null)
                this.rawResponse.headers = new HashMap<>() {{
                    put("Connection", "Closed");
                }};
            this.rawResponse.headers.put(name, value);
            return this;
        }

        public Builder body(String body) {
            this.rawResponse.body = body;
            return this;
        }

        public Response build() {
            return this.rawResponse;
        }
    }

    @Override
    public String toString() {
        StringBuilder result = new StringBuilder();
        if (this.status == null) {
            result.append(this.stringStatus);
        } else {
            result.append(this.version).append(' ');
            result.append(this.status.toString());
        }
        if (this.headers != null) {
            result.append('\n');
            this.headers.forEach((key, value) -> {
                result.append(key)
                        .append(": ")
                        .append(value)
                        .append('\n');
            });
        }
        if (this.body != null) {
            result.append('\n');
            result.append(this.body);
        }
        return result.toString();
    }
}
