package com.litehost.core;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.Socket;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

class Server extends Thread {
    Listener listener;
    Wrapper wrapper;
    Respondent respondent;
    BlockingQueue<ClientRequest> tasks;
    BlockingQueue<Socket> connections;
    BlockingQueue<Future<ClientRequest>> finishedTasks;
    private ExecutorService scheduler;
    private static int maxThreadPool;
    final String domainName;
    final static Object shutdownLock = new Object();
    final int port;

    Server(int port, int minThreadPool, int maxThreadPool, String domainName) {
        ExceptionHandlers.initialize(this);
        Server.maxThreadPool = maxThreadPool;
        this.port = port;
        this.domainName = domainName;
        this.connections = new LinkedBlockingQueue<>();
        this.tasks = new LinkedBlockingQueue<>();
        this.finishedTasks = new LinkedBlockingQueue<>();
        this.scheduler = new ThreadPoolExecutor(minThreadPool, maxThreadPool,
                5, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<>(),
                runnable -> {
                    Thread thread = new Thread(runnable);
                    thread.setUncaughtExceptionHandler(ExceptionHandlers.logHandler());
                    return thread;
                });
        this.wrapper = new Wrapper(this.connections, this.tasks, maxThreadPool);
        this.listener = new Listener(this.port, this.connections, domainName);
        this.respondent = new Respondent(this.finishedTasks);
    }

    static int getMaxThreadPool() {
        return maxThreadPool;
    }


    @Override
    public void start() {
        this.setDaemon(true);
        this.setUncaughtExceptionHandler(ExceptionHandlers.stopHandler());
        long zeroTime = System.nanoTime();
        System.out.println(" Server staring up...");
        this.listener.start();
        this.wrapper.start();
        this.respondent.start();
        super.start();
        System.out.println(" Finished starting up, elapsed time: " + (System.nanoTime() - zeroTime) / 1_000_000 + "ms");
        System.out.println(" Listening on port " + this.listener.getPort());
        Logger logger = LoggerFactory.getLogger(this.getClass());
        logger.info(" Server successfully started");
    }


    public void stopServer() {
        this.interrupt();
    }

    private void runShutdownProcedure() {
        this.wrapper.interrupt();
        try {
            synchronized (shutdownLock) {
                shutdownLock.wait();
            }
            while (!this.tasks.isEmpty())
                this.finishedTasks.put(this.scheduler.submit(this.tasks.poll()));
            synchronized (shutdownLock) {
                shutdownLock.notify();
            }
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        this.respondent.interrupt();
    }

    @Override
    public void run() {
        while (true) {
            try {
                if (interrupted()) {
                    this.runShutdownProcedure();
                    return;
                }
                ClientRequest clientRequest = this.tasks.take();
                Future<ClientRequest> runningTask = this.scheduler.submit(clientRequest);
                this.finishedTasks.put(runningTask);
            } catch (InterruptedException e) {
                this.runShutdownProcedure();
                return;
            }
        }
    }
}
