package com.litehost.core;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

public class Respondent extends Thread {
    private BlockingQueue<Future<ClientRequest>> finishedTasks;
    private Logger logger;


    Respondent(BlockingQueue<Future<ClientRequest>> finishedTasks) {
        this.finishedTasks = finishedTasks;
        this.logger = LoggerFactory.getLogger(this.getClass());
    }

    @Override
    public synchronized void start() {
        this.setDaemon(false);
        this.setUncaughtExceptionHandler(ExceptionHandlers.stopHandler());
        super.start();
    }

    private void finalizeRequest(Future<ClientRequest> processingRequest) throws IOException {
        try (ClientRequest clientRequest = processingRequest.get()) {
            HostController.OUTPUT_PORT.acquire();
            clientRequest.sendResponse();
        } catch (InterruptedException | ExecutionException | IOException e) {
            this.logger.debug(e.getMessage());
        } finally {
            HostController.OUTPUT_PORT.release();
        }
    }

    private void runShutdownProcedure() {
        while (!this.finishedTasks.isEmpty()) {
            try {
                this.finalizeRequest(this.finishedTasks.poll());
            } catch (IOException e) {
                this.logger.debug(e.getMessage());
            }
        }
        this.logger.info("LiteHost shuts down");
    }

    @Override
    public void run() {
        while (true) {
            try {
                if (interrupted()) {
                    this.runShutdownProcedure();
                    return;
                }
                Future<ClientRequest> processingRequest = this.finishedTasks.take();
                this.finalizeRequest(processingRequest);
            } catch (InterruptedException e) {
                this.runShutdownProcedure();
                return;
            } catch (IOException e) {
                this.logger.debug(e.getMessage());
            }
        }
    }
}
