package com.litehost.core;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ExceptionHandlers {
    private ExceptionHandlers() {
    }

    private static boolean initialized = false;

    static void initialize(Server server) {
        if (!initialized) {
            ExceptionHandlers.server = server;
            initialized = true;
        }
    }

    private static Server server;

    static Thread.UncaughtExceptionHandler logHandler() {
        return new LogHandler();
    }

    static Thread.UncaughtExceptionHandler stopHandler() {
        return (t, e) -> {
            logHandler().uncaughtException(t, e);
            System.exit(420);
        };
    }

    static Thread.UncaughtExceptionHandler retryHandler(int maxRetry) {
        return new RetryHandler(maxRetry);
    }

    private static class LogHandler implements Thread.UncaughtExceptionHandler {
        private String additionalMessage = null;

        LogHandler() {
        }

        public void additionalMessage(String msg) {
            this.additionalMessage = msg;
        }

        @Override
        public void uncaughtException(Thread t, Throwable e) {
            StringBuilder cause = new StringBuilder(e.getClass().getSimpleName());
            cause.append(' ');
            cause.append(e.getStackTrace()[0].toString());
            Logger logger = LoggerFactory.getLogger(Server.class);
            if (this.additionalMessage != null)
                logger.debug("Exception " + cause.toString() + "at " + t.getName() + ", information: " + this.additionalMessage);
            else logger.debug("Exception " + cause.toString() + "at " + t.getName());
        }
    }

    private static class RetryHandler implements Thread.UncaughtExceptionHandler {
        private int retryCounter;
        private int maxRetry;
        private LogHandler logHandler;

        RetryHandler(int maxRetry) {
            this.maxRetry = maxRetry;
            this.retryCounter = 0;
            this.logHandler = ( LogHandler ) ExceptionHandlers.logHandler();
        }

        private void registerRetry() {
            this.retryCounter++;
        }

        private boolean validRetry() {
            this.registerRetry();
            return this.retryCounter <= this.maxRetry;
        }

        private Thread getNewThread(String threadName) {
            Thread newThread;
            switch (threadName) {
                case "Listener":
                    server.listener = new Listener(server.port, server.connections, server.domainName);
                    newThread = server.listener;
                    break;
                case "Respondent":
                    server.respondent = new Respondent(server.finishedTasks);
                    newThread = server.respondent;
                    break;
                case "Wrapper":
                    server.wrapper = new Wrapper(server.connections, server.tasks, Server.getMaxThreadPool());
                    newThread = server.wrapper;
                    break;
                default:
                    throw new IllegalStateException("Unexpected value: " + threadName);
            }
            return newThread;
        }

        @Override
        public void uncaughtException(Thread t, Throwable e) {
            if (this.validRetry()) {
                this.logHandler.uncaughtException(t, e);
                Thread newThread = this.getNewThread(t.getClass().getSimpleName());
                newThread.start();
            } else {
                this.logHandler.additionalMessage("Cannot retry anymore!");
                this.logHandler.uncaughtException(t, e);
            }
        }
    }
}
