package io.gitlab.mguimard.openrgb.utils;

import io.gitlab.mguimard.openrgb.entity.OpenRGBColor;

/**
 * Color manipulation utility class
 */
public class ColorUtils {

    /**
     * Generate an array of OpenRGBColor using HSV, see https://en.wikipedia.org/wiki/Hue#/media/File:Hsv-hues-cf-lch-hues.png
     * <p>
     * hue angle	color code	color name
     * 0°	        #FF0000	    red
     * 15°	        #FF4000	    vermilion
     * 30°	        #FF8000	    orange
     * 45°	        #FFBF00	    golden yellow
     * 60°	        #FFFF00	    yellow
     * 75°	        #BFFF00	    yellowish green
     * 90°	        #80FF00	    yellowish green, chartreuse
     * 105°	        #40FF00	    leaf green
     * 120°	        #00FF00	    green
     * 135°	        #00FF40	    cobalt green
     * 150°	        #00FF80	    emerald green
     * 165°	        #00FFBF	    turquoise green
     * 180°	        #00FFFF	    turquoise blue
     * 195°	        #00BFFF	    cerulean blue
     * 210°	        #0080FF	    azure
     * 225°	        #0040FF	    blue, cobalt blue
     * 240°	        #0000FF	    blue
     * 255°	        #4000FF	    hyacinth
     * 270°	        #8000FF	    violet
     * 285°	        #BF00FF	    purple
     * 300°	        #FF00FF	    magenta
     * 315°	        #FF00BF	    reddish purple
     * 330°	        #FF0080	    ruby red, crimson
     * 345°	        #FF0040	    carmine
     *
     * @param size  palette size
     * @param start start color (0-360)
     * @param end   end color (0-360)
     * @return the rainbow array
     */
    public static OpenRGBColor[] generateRainbow(int size, int start, int end) {

        if (size <= 0) {
            return new OpenRGBColor[0];
        }

        start = Math.min(Math.max(start, 0), 360);
        end = Math.max(Math.min(end, 360), 0);

        if (start > end) {
            int tmp = start;
            start = end;
            end = tmp;
        }

        System.out.println(start);
        System.out.println(end);

        double step = (end - start) / (size * 1.0);

        OpenRGBColor[] colors = new OpenRGBColor[size];

        for (int i = 0; i < colors.length; i++) {
            float hue = (float) (start + step * i);
            System.out.println(hue);
            colors[i] = OpenRGBColor.fromHSB(hue, 1.0f, 1.0f);
        }

        return colors;
    }

    /**
     * Get a reverse copy from a given OpenRGBColor array
     *
     * @param colors the original array
     * @return the reversed copy
     */
    public static OpenRGBColor[] copyAndReverse(OpenRGBColor[] colors) {
        OpenRGBColor[] reverse = new OpenRGBColor[colors.length];

        if (colors.length > 0) {
            int count = colors.length - 1;
            for (OpenRGBColor color : colors) {
                reverse[count] = color;
                count--;
            }
        }

        return reverse;
    }


}
