package io.gitlab.mguimard.openrgb.utils;

import io.gitlab.mguimard.openrgb.entity.*;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;

public class DeviceParser {

    public static OpenRGBDevice from(ByteBuffer buffer, int serverProtocol) throws IOException {
        OpenRGBDevice device = new OpenRGBDevice();
        String s = new String(buffer.array(), StandardCharsets.US_ASCII);

        ByteArrayInputStream byteArrayInputStream =
                new ByteArrayInputStream(buffer.array());
        LittleEndianInputStream input = new LittleEndianInputStream(byteArrayInputStream);

        int duplicatePacketLength = input.readInt();

        device.setType(input.readInt());
        device.setName(input.readAscii());
        device.setVendor(serverProtocol >= 1 ? input.readAscii() : null);
        device.setDesc(input.readAscii());
        device.setVersion(input.readAscii());
        device.setSerial(input.readAscii());
        device.setLocation(input.readAscii());

        int modeCount = input.readUnsignedShort();
        device.setActiveMode(input.readInt());
        device.setModes(readModes(input, modeCount));

        int zoneCount = input.readUnsignedShort();
        device.setZones(readZones(input, zoneCount));

        int ledCount = input.readUnsignedShort();
        device.setLeds(readLeds(input, ledCount));

        int colorCount = input.readUnsignedShort();
        device.setColors(readColors(input, colorCount));

        return device;
    }

    private static List<OpenRGBMode> readModes(LittleEndianInputStream input, int modeCount) throws IOException {

        List<OpenRGBMode> openRGBModes = new ArrayList<>();

        for (int modeIndex = 0; modeIndex < modeCount; modeIndex++) {
            String modeName = input.readAscii();

            int value = input.readInt();
            int flags = input.readInt();

            FlagsCheck flagsCheck = new FlagsCheck(flags);

            int speedMin = 0;
            int speedMax = 0;

            if (flagsCheck.hasSpeed()) {
                speedMin = input.readInt();
                speedMax = input.readInt();
            } else {
                input.skip(8);
            }

            int colorMin = 0;
            int colorMax = 0;

            if (flagsCheck.hasModeSpecificColor()) {
                colorMin = input.readInt();
                colorMax = input.readInt();
            } else {
                input.skip(8);
            }

            int speed = 0;
            if (flagsCheck.hasSpeed()) {
                speed = input.readInt();
            } else {
                input.skip(4);
            }

            int direction = 0;
            if (flagsCheck.hasDirection()) {
                direction = input.readInt();
            } else {
                input.skip(4);
            }

            int colorMode = input.readInt();
            int colorLength = input.readUnsignedShort();

            List<OpenRGBColor> openRGBColors = new ArrayList<>();

            for (int colorIndex = 0; colorIndex < colorLength; colorIndex++) {
                OpenRGBColor openRGBColor = readColor(input);
                openRGBColors.add(openRGBColor);
            }

            openRGBModes.add(new OpenRGBMode(modeName, value, flags, speedMin, speedMax, colorMin, colorMax, speed, direction, colorMode, openRGBColors));
        }

        return openRGBModes;
    }

    private static List<OpenRGBZone> readZones(LittleEndianInputStream input, int zoneCount) throws IOException {
        List<OpenRGBZone> openRGBZones = new ArrayList<>();

        for (int zoneIndex = 0; zoneIndex < zoneCount; zoneIndex++) {
            String zoneName = input.readAscii();
            int type = input.readInt();
            int ledsMin = input.readInt();
            int ledsMax = input.readInt();
            int ledsCount = input.readInt();

            int matrixSize = input.readUnsignedShort();
            int height = 0;
            int width = 0;

            if (matrixSize > 0) {
                height = input.readInt();
                width = input.readInt();
            }

            int[][] matrix = new int[height][width];

            for (int i = 0; i < height; i++) {
                for (int j = 0; j < width; j++) {
                    matrix[i][j] = input.readInt();
                }
            }

            openRGBZones.add(new OpenRGBZone(
                    zoneName, type, ledsMin, ledsMax, ledsCount, height, width, matrix));
        }

        return openRGBZones;
    }

    private static List<OpenRGBLed> readLeds(LittleEndianInputStream input, int ledCount) throws IOException {
        List<OpenRGBLed> leds = new ArrayList<>();
        for (int ledIndex = 0; ledIndex < ledCount; ledIndex++) {
            String ledName = input.readAscii();
            OpenRGBColor openRGBColor = readColor(input);
            leds.add(new OpenRGBLed(ledName, openRGBColor));
        }
        return leds;
    }

    private static List<OpenRGBColor> readColors(LittleEndianInputStream input, int colorCount) throws IOException {
        List<OpenRGBColor> colors = new ArrayList<>();
        for (int colorIndex = 0; colorIndex < colorCount; colorIndex++) {
            colors.add(readColor(input));
        }
        return colors;
    }

    private static OpenRGBColor readColor(LittleEndianInputStream input) throws IOException {
        OpenRGBColor color = new OpenRGBColor(
                (byte) input.read(),
                (byte) input.read(),
                (byte) input.read());
        input.skip(1);
        return color;
    }

    static class FlagsCheck {
        int flags;

        FlagsCheck(int flags) {
            this.flags = flags;
        }

        boolean hasSpeed() {
            return (flags & 1) != 0;
        }

        boolean hasDirectionLR() {
            return checkBit(1);
        }

        boolean hasDirectionUP() {
            return checkBit(2);
        }

        boolean hasDirectionHv() {
            return checkBit(3);
        }

        boolean hasBrightness() {
            return checkBit(4);
        }

        boolean hasPerLedColor() {
            return checkBit(5);
        }

        boolean hasModeSpecificColor() {
            return checkBit(6);
        }

        boolean hasRandomColor() {
            return checkBit(7);
        }

        boolean hasDirection() {
            return (flags & 0b0000_1110) != 0;
        }

        boolean checkBit(int bit) {
            return ((flags & 1) << bit) != 0;
        }
    }
}
