package io.gitlab.mguimard.openrgb.client;

import io.gitlab.mguimard.openrgb.entity.OpenRGBColor;
import io.gitlab.mguimard.openrgb.entity.OpenRGBDevice;
import io.gitlab.mguimard.openrgb.entity.OpenRGBMode;
import io.gitlab.mguimard.openrgb.utils.DeviceParser;
import io.gitlab.mguimard.openrgb.utils.Utils;

import java.io.*;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.logging.Logger;

/**
 *
 */
public class OpenRGBClient {

    private static final Logger LOGGER = Logger.getLogger(OpenRGBClient.class.getName());
    private static final int HEADER_SIZE = 16;
    private static final byte[] MAGIC_STRING = "ORGB".getBytes(StandardCharsets.US_ASCII);

    private String host;
    private int port;
    private String name;
    private OutputStream out;
    private InputStream in;
    private Socket socket;
    private int serverProtocol;

    /**
     * Returns an instance of OpenRGBClient
     *
     * @param host server ip or host
     * @param port server port
     * @param name client name displayed in OpenRGBServer
     */
    public OpenRGBClient(String host, int port, String name) {
        this.host = host;
        this.port = port;
        this.name = name;
    }

    /**
     * Connect to the OpenRGBServer
     *
     * @throws IOException
     */
    public void connect() throws IOException {
        disconnect();
        socket = new Socket();
        socket.connect(new InetSocketAddress(host, port));
        this.out = new BufferedOutputStream(socket.getOutputStream());
        this.in = new BufferedInputStream(socket.getInputStream());
        this.setClientName();
        this.serverProtocol = this.getProtocolVersion();
    }

    /**
     * Disconnect the socket
     *
     * @throws IOException
     */
    public void disconnect() throws IOException {
        if (null != socket && !socket.isClosed()) {
            socket.close();
        }
    }

    public int getProtocolVersion() throws IOException {
        byte[] data = Utils.byteArrayFromInt(serverProtocol);
        sendMessage(OpenRGBCommand.GetProtocolVersion, data);
        return read().getInt();
    }

    /**
     * Send the client name to the OpenRGBServer
     *
     * @throws IOException
     */
    public void setClientName() throws IOException {
        sendMessage(OpenRGBCommand.SetClientName, (name + '\u0000').getBytes(StandardCharsets.US_ASCII));
    }

    /**
     * Retrieve the count of controllers discovered by OpenRGBServer
     *
     * @return the count of controllers
     * @throws IOException
     */
    public int getControllerCount() throws IOException {
        sendMessage(OpenRGBCommand.RequestControllerCount);
        return read().getInt();
    }

    /**
     * Returns all controller information for a given deviceId
     *
     * @param deviceId the device id sent by OpenRGBServer
     * @return the controller information
     * @throws IOException
     */
    public OpenRGBDevice getDeviceController(int deviceId) throws IOException {
        byte[] data = Utils.byteArrayFromInt(serverProtocol);
        sendMessage(OpenRGBCommand.RequestControllerData, deviceId, data);
        return DeviceParser.from(read(), this.serverProtocol);
    }

    /**
     * @param deviceId
     * @param ledIndex
     * @param color
     * @throws IOException
     */
    public void updateLed(int deviceId, int ledIndex, OpenRGBColor color) throws IOException {
        ByteBuffer data = ByteBuffer.allocate(8)
                .order(ByteOrder.LITTLE_ENDIAN)
                .putInt(ledIndex)
                .put(color.getRed())
                .put(color.getGreen())
                .put(color.getBlue());
        data.position(data.position() + 1);
        sendMessage(OpenRGBCommand.UpdateSingleLed, deviceId, data.array());
    }


    /**
     * @param deviceId
     * @param colors
     * @throws IOException
     */
    public void updateLeds(int deviceId, OpenRGBColor[] colors) throws IOException {
        ByteBuffer data = (ByteBuffer) ByteBuffer.allocate(4 + 2 + 4 * colors.length)
                .order(ByteOrder.LITTLE_ENDIAN)
                .position(4);
        data.putShort(((short) colors.length));

        for (int i = 0, colorsLength = colors.length; i < colorsLength; i++) {
            OpenRGBColor color = colors[i];
            data
                    .put(color.getRed())
                    .put(color.getGreen())
                    .put(color.getBlue())
                    .position(data.position() + 1);
        }

        sendMessage(OpenRGBCommand.UpdateLeds, deviceId, data.array());
    }

    /**
     * @param deviceId
     * @param zoneId
     * @param colors
     * @throws IOException
     */
    public void updateZoneLeds(int deviceId, int zoneId, OpenRGBColor[] colors) throws IOException {
        ByteBuffer data = (ByteBuffer) ByteBuffer.allocate(4 + 4 + 2 + 4 * colors.length)
                .order(ByteOrder.LITTLE_ENDIAN)
                .position(4);
        data.putInt(zoneId);
        data.putShort(((short) colors.length));

        for (int i = 0, colorsLength = colors.length; i < colorsLength; i++) {
            OpenRGBColor color = colors[i];
            data
                    .put(color.getRed())
                    .put(color.getGreen())
                    .put(color.getBlue())
                    .position(data.position() + 1);
        }

        sendMessage(OpenRGBCommand.UpdateZoneLeds, deviceId, data.array());
    }


    /**
     * Change the mode for a controller
     *
     * @param deviceId  the device id sent by OpenRGBServer
     * @param modeIndex the mode to set
     * @throws IOException
     */
    public void updateMode(int deviceId, int modeIndex, OpenRGBMode mode) throws IOException {
        List<OpenRGBColor> colors = mode.getColors();
        byte[] nameBytes = mode.getName().getBytes(StandardCharsets.US_ASCII);

        ByteBuffer data = ByteBuffer.allocate(4 + 4 + 2 + nameBytes.length + 4 + 4 + 4 + 4 + 4 + 4 + 4 + 4 + 4 + 2 + 4 * colors.size())
                .order(ByteOrder.LITTLE_ENDIAN);

        data.position(4);

        data
                .putInt(modeIndex)
                .putShort((short) nameBytes.length)
                .put(nameBytes)
                .putInt(mode.getValue())
                .putInt(mode.getFlags())
                .putInt(mode.getSpeedMin())
                .putInt(mode.getSpeedMax())
                .putInt(mode.getColorMin())
                .putInt(mode.getColorMax())
                .putInt(mode.getSpeed())
                .putInt(mode.getDirection())
                .putInt(mode.getColorMode())
                .putShort((short) colors.size());

        for (OpenRGBColor color : colors) {
            data
                    .put(color.getRed())
                    .put(color.getGreen())
                    .put(color.getBlue())
                    .position(data.position() + 1);
        }
        sendMessage(OpenRGBCommand.UpdateMode, deviceId, data.array());
    }

    private byte[] createMagicHeader(int deviceId, OpenRGBCommand openRGBCommand, int length) {
        ByteBuffer byteBuffer = ByteBuffer
                .allocate(HEADER_SIZE)
                .order(ByteOrder.LITTLE_ENDIAN)
                .put(MAGIC_STRING)
                .putInt(deviceId)
                .putInt(openRGBCommand.getValue())
                .putInt(length);
        return byteBuffer.array();
    }

    private void sendMessage(OpenRGBCommand command) throws IOException {
        sendMessage(command, 0, null);
    }

    private void sendMessage(OpenRGBCommand command, int deviceId) throws IOException {
        sendMessage(command, deviceId, null);
    }

    private void sendMessage(OpenRGBCommand command, byte[] data) throws IOException {
        sendMessage(command, 0, data);
    }

    private void sendMessage(OpenRGBCommand command, int deviceId, byte[] data) throws IOException {
        byte[] header = createMagicHeader(
                deviceId, command, null != data ? data.length : 0);

        out.write(header);
        if (null != data && data.length > 0) {
            out.write(data);
        }
        out.flush();
    }

    private ByteBuffer read() throws IOException {
        byte[] header = new byte[HEADER_SIZE];
        in.read(header, 0, HEADER_SIZE);

        ByteBuffer byteBuffer = ByteBuffer.wrap(header).order(ByteOrder.LITTLE_ENDIAN);
        //int deviceId = byteBuffer.getInt(4);
        //int commandId = byteBuffer.getInt(8);
        int length = byteBuffer.getInt(12);

        // read packet
        byte[] packet = new byte[length];
        in.read(packet, 0, length);

        LOGGER.fine("ResHeader=<" + Utils.bytesToHex(header) + ">");
        LOGGER.fine("ResPacket=<" + Utils.bytesToHex(packet) + ">");

        return ByteBuffer.wrap(packet).order(ByteOrder.LITTLE_ENDIAN);
    }


}
