package io.gitlab.guiVista.core

import glib2.*
import kotlinx.cinterop.reinterpret

public actual class Object private constructor(ptr: gpointer?) : ObjectBase {
    public val objPtr: gpointer? = ptr

    /**
     * Creates a binding between [sourceProperty] on this Object and [targetProperty] on [targetObj]. Whenever the
     * [sourceProperty] is changed the [targetProperty] is updated using the same value.
     *
     * The binding will automatically be removed when either the source or the target objects are finalized. To remove
     * the binding without affecting the source, and the target objects you can just call [DataBinding.close] on the
     * returned [DataBinding] instance.
     * @param sourceProperty The property on this [Object] to bind.
     * @param targetObj The target Object.
     * @param targetProperty The property on the [targetObj] to bind.
     * @param biDirectional If set to *true* then a change made to [sourceProperty] updates [targetProperty], and vice
     * versa.
     */
    public fun bindProperty(
        sourceProperty: String,
        targetObj: gpointer?,
        targetProperty: String,
        biDirectional: Boolean = false
    ): DataBinding {
        val tmp = g_object_bind_property(
            source = objPtr,
            source_property = sourceProperty,
            target = targetObj,
            target_property = targetProperty,
            flags = if (biDirectional) G_BINDING_BIDIRECTIONAL else G_BINDING_DEFAULT
        )
        return DataBinding.fromPointer(tmp)
    }

    public actual fun fetchProperty(name: String): ValueBase {
        val result = Value.create()
        g_object_get_property(`object` = objPtr?.reinterpret(), property_name = name, value = result.gValue)
        return result
    }

    public actual fun changeBooleanProperty(name: String, value: Boolean) {
        val valueObj = Value.create().apply { changeBoolean(value) }
        g_object_set_property(`object` = objPtr?.reinterpret(), property_name = name, value = valueObj.gValue)
        valueObj.close()
    }

    public actual fun changeStaticCharProperty(name: String, value: Char) {
        val valueObj = Value.create().apply { changeStaticChar(value) }
        g_object_set_property(`object` = objPtr?.reinterpret(), property_name = name, value = valueObj.gValue)
        valueObj.close()
    }

    public actual fun changeDoubleProperty(name: String, value: Double) {
        val valueObj = Value.create().apply { changeDouble(value) }
        g_object_set_property(`object` = objPtr?.reinterpret(), property_name = name, value = valueObj.gValue)
        valueObj.close()
    }

    public actual fun changeFloatProperty(name: String, value: Float) {
        val valueObj = Value.create().apply { changeFloat(value) }
        g_object_set_property(`object` = objPtr?.reinterpret(), property_name = name, value = valueObj.gValue)
        valueObj.close()
    }

    public actual fun changeGTypeProperty(name: String, value: ULong) {
        val valueObj = Value.create().apply { changeGType(value) }
        g_object_set_property(`object` = objPtr?.reinterpret(), property_name = name, value = valueObj.gValue)
        valueObj.close()
    }

    public actual fun changeIntProperty(name: String, value: Int) {
        val valueObj = Value.create().apply { changeInt(value) }
        g_object_set_property(`object` = objPtr?.reinterpret(), property_name = name, value = valueObj.gValue)
        valueObj.close()
    }

    public actual fun changeLongProperty(name: String, value: Long) {
        val valueObj = Value.create().apply { changeLong(value) }
        g_object_set_property(`object` = objPtr?.reinterpret(), property_name = name, value = valueObj.gValue)
        valueObj.close()
    }

    public actual fun changeObjectProperty(name: String, value: Object?) {
        val valueObj = Value.create().apply { changeObject(value) }
        g_object_set_property(`object` = objPtr?.reinterpret(), property_name = name, value = valueObj.gValue)
        valueObj.close()
    }

    public actual fun changeUByteProperty(name: String, value: UByte) {
        val valueObj = Value.create().apply { changeUByte(value) }
        g_object_set_property(`object` = objPtr?.reinterpret(), property_name = name, value = valueObj.gValue)
        valueObj.close()
    }

    public actual fun changeUIntProperty(name: String, value: UInt) {
        val valueObj = Value.create().apply { changeUInt(value) }
        g_object_set_property(`object` = objPtr?.reinterpret(), property_name = name, value = valueObj.gValue)
        valueObj.close()
    }

    public actual fun changeULongProperty(name: String, value: ULong) {
        val valueObj = Value.create().apply { changeULong(value) }
        g_object_set_property(`object` = objPtr?.reinterpret(), property_name = name, value = valueObj.gValue)
        valueObj.close()
    }

    public actual fun changeStaticStringProperty(name: String, value: String) {
        val valueObj = Value.create().apply { changeStaticString(value) }
        g_object_set_property(`object` = objPtr?.reinterpret(), property_name = name, value = valueObj.gValue)
    }

    public companion object {
        public fun fromPointer(ptr: gpointer?): Object = Object(ptr)
    }

    override fun close() {
        g_object_unref(objPtr)
    }
}
